/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.phoenix.iterate;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.sql.SQLException;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;

import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.HTableInterface;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.WritableUtils;
import org.apache.phoenix.compile.StatementContext;
import org.apache.phoenix.coprocessor.BaseScannerRegionObserver;
import org.apache.phoenix.execute.TupleProjector;
import org.apache.phoenix.expression.Expression;
import org.apache.phoenix.expression.ExpressionType;
import org.apache.phoenix.query.KeyRange;
import org.apache.phoenix.schema.tuple.ResultTuple;
import org.apache.phoenix.schema.tuple.Tuple;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;

import static org.apache.phoenix.coprocessor.BaseScannerRegionObserver.FILTER_PB_BYTES;

public class GettingResultIterator implements ResultIterator {
    private static final Logger logger = LoggerFactory.getLogger(GettingResultIterator.class);
    private StatementContext context;
    private HTableInterface hTable = null;
    private List<Get> gets;
    private Iterator<Result> rsIterator = null;
    private final Expression where;

    public GettingResultIterator(StatementContext context, Expression where) {
        this.context = context;
        this.where = where;
        this.gets = Lists.newArrayListWithExpectedSize(context.getScanRanges().getPointLookupCount());

        Scan scan = context.getScan();
        List<List<KeyRange>> ranges = context.getScanRanges().getRanges();
        for (KeyRange range : ranges.get(0)) {
            Get get = new Get(range.getLowerRange());
            for (Map.Entry<byte[], NavigableSet<byte[]>> entry : scan.getFamilyMap().entrySet()) {
                byte[] family = entry.getKey();
                if (entry.getValue() != null) {
                    for (byte[] q : entry.getValue()) {
                        get.addColumn(family, q);
                    }
                } else {
                    get.addFamily(family);
                }
            }

            for (Map.Entry<String, byte[]> attr : scan.getAttributesMap().entrySet()) {
                get.setAttribute(attr.getKey(), attr.getValue());
            }

            setGetAttribute(get);
            gets.add(get);
        }
    }

    private void setGetAttribute(Get get) {
        get.setAttribute(BaseScannerRegionObserver.USE_NEW_VALUE_COLUMN_QUALIFIER, Bytes.toBytes(true));
        TupleProjector.copySerializeProjectorIntoGetFromScan(context.getScan(), get);
        get.setAttribute(BaseScannerRegionObserver.QUALIFIER_ENCODING_SCHEME,
                new byte[]{context.getCurrentTable().getTable().getEncodingScheme().getSerializedMetadataValue()});
        if (where != null) {
            serializeExpressionIntoGet(get, FILTER_PB_BYTES, where);
        }
    }

    private static void serializeExpressionIntoGet(Get get, String attrName, Expression expression) {
        ByteArrayOutputStream stream = new ByteArrayOutputStream();
        try {
            DataOutputStream output = new DataOutputStream(stream);
            WritableUtils.writeVInt(output, ExpressionType.valueOf(expression).ordinal());
            expression.write(output);
            get.setAttribute(attrName, stream.toByteArray());
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally {
            try {
                stream.close();
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    @Override
    public Tuple next() throws SQLException {
        try {
            if (hTable == null && rsIterator == null) {
                hTable = context.getConnection().getQueryServices()
                        .getTable(context.getCurrentTable().getTable().getPhysicalName().getBytes());
                rsIterator = Iterators.forArray(hTable.get(gets));
            }

            Result r = null;
            while (rsIterator.hasNext()) {
                r = rsIterator.next();
                if (!r.isEmpty()) {
                    break;
                }
            }
            if (r == null) {
                return null;
            } else {
                return new ResultTuple(r);
            }
        } catch (IOException e) {
            throw new SQLException(e);
        }
    }

    @Override
    public void explain(List<String> planSteps) {
        String step = "\n GETS " + gets.size() + " KEYS";
        planSteps.add(step);
    }

    @Override
    public void close() throws SQLException {
        // do nothing
    }
}
