/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.phoenix.mapreduce.util;

import com.aliyun.odps.Odps;
import com.aliyun.odps.PartitionSpec;
import com.aliyun.odps.account.AliyunAccount;
import com.aliyun.odps.tunnel.TableTunnel;
import com.aliyun.odps.tunnel.TunnelException;
import com.google.common.base.Function;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.io.WritableUtils;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.mapreduce.Mapper;
import org.apache.hadoop.mapreduce.lib.db.DBWritable;
import org.apache.phoenix.jdbc.PhoenixConnection;
import org.apache.phoenix.mapreduce.ODPSMapper;
import org.apache.phoenix.mapreduce.PhoenixInputFormat;
import org.apache.phoenix.mapreduce.PhoenixOutputFormat;
import org.apache.phoenix.mapreduce.bulkload.TableRowkeyPair;
import org.apache.phoenix.mapreduce.util.PhoenixConfigurationUtil.SchemaType;
import org.apache.phoenix.query.QueryConstants;
import org.apache.phoenix.schema.PColumn;
import org.apache.phoenix.schema.PColumnFamily;
import org.apache.phoenix.schema.PTable;
import org.apache.phoenix.util.ColumnInfo;
import org.apache.phoenix.util.EncodedColumnsUtil;
import org.apache.phoenix.util.PhoenixRuntime;
import org.apache.phoenix.util.SchemaUtil;

import javax.annotation.Nullable;
import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.sql.SQLException;
import java.util.*;

/**
 * Utility class for setting Configuration parameters for the Map Reduce job
 */
public final class PhoenixMapReduceUtil {

    private PhoenixMapReduceUtil() {

    }

    /**
     *
     * @param job
     * @param inputClass DBWritable class
     * @param tableName  Input table name
     * @param conditions Condition clause to be added to the WHERE clause. Can be <tt>null</tt> if there are no conditions.
     * @param fieldNames fields being projected for the SELECT query.
     */
    public static void setInput(final Job job, final Class<? extends DBWritable> inputClass, final String tableName,
                                final String conditions, final String... fieldNames) {
        final Configuration configuration = setInput(job, inputClass, tableName);
        if(conditions != null) {
            PhoenixConfigurationUtil.setInputTableConditions(configuration, conditions);
        }
        PhoenixConfigurationUtil.setSelectColumnNames(configuration, fieldNames);
    }

    /**
     *
     * @param job
     * @param inputClass  DBWritable class
     * @param tableName   Input table name
     * @param inputQuery  Select query.
     */
    public static void setInput(final Job job, final Class<? extends DBWritable> inputClass, final String tableName, final String inputQuery) {
        final Configuration configuration = setInput(job, inputClass, tableName);
        PhoenixConfigurationUtil.setInputQuery(configuration, inputQuery);
        PhoenixConfigurationUtil.setSchemaType(configuration, SchemaType.QUERY);
     }

    /**
     *
     * @param job
     * @param inputClass DBWritable class
     * @param snapshotName The name of a snapshot (of a table) to read from
     * @param tableName Input table name
     * @param restoreDir a temporary dir to copy the snapshot files into
     * @param conditions Condition clause to be added to the WHERE clause. Can be <tt>null</tt> if there are no conditions.
     * @param fieldNames fields being projected for the SELECT query.
     */
    public static void setInput(final Job job, final Class<? extends DBWritable> inputClass, final String snapshotName, String tableName,
        Path restoreDir, final String conditions, final String... fieldNames) throws
        IOException {
        final Configuration configuration = setSnapshotInput(job, inputClass, snapshotName, tableName, restoreDir);
        if(conditions != null) {
            PhoenixConfigurationUtil.setInputTableConditions(configuration, conditions);
        }
        PhoenixConfigurationUtil.setSelectColumnNames(configuration, fieldNames);
    }

    /**
     *
     * @param job
     * @param inputClass DBWritable class
     * @param snapshotName The name of a snapshot (of a table) to read from
     * @param tableName Input table name
     * @param restoreDir a temporary dir to copy the snapshot files into
     * @param inputQuery The select query
     */
    public static void setInput(final Job job, final Class<? extends DBWritable> inputClass, final String snapshotName, String tableName,
        Path restoreDir, String inputQuery) throws
        IOException {
        final Configuration configuration = setSnapshotInput(job, inputClass, snapshotName, tableName, restoreDir);
        if(inputQuery != null) {
            PhoenixConfigurationUtil.setInputQuery(configuration, inputQuery);
        }

    }

    /**
     *
     * @param job
     * @param inputClass DBWritable class
     * @param snapshotName The name of a snapshot (of a table) to read from
     * @param tableName Input table name
     * @param restoreDir a temporary dir to copy the snapshot files into
     */
    private static Configuration setSnapshotInput(Job job, Class<? extends DBWritable> inputClass, String snapshotName,
        String tableName, Path restoreDir) {
        job.setInputFormatClass(PhoenixInputFormat.class);
        final Configuration configuration = job.getConfiguration();
        PhoenixConfigurationUtil.setInputClass(configuration, inputClass);
        PhoenixConfigurationUtil.setSnapshotNameKey(configuration, snapshotName);
        PhoenixConfigurationUtil.setInputTableName(configuration, tableName);

        PhoenixConfigurationUtil.setRestoreDirKey(configuration, new Path(restoreDir, UUID.randomUUID().toString()).toString());
        PhoenixConfigurationUtil.setSchemaType(configuration, SchemaType.QUERY);
        return configuration;
    }

    private static Configuration setInput(final Job job, final Class<? extends DBWritable> inputClass, final String tableName){
        job.setInputFormatClass(PhoenixInputFormat.class);
        final Configuration configuration = job.getConfiguration();
        PhoenixConfigurationUtil.setInputTableName(configuration, tableName);
        PhoenixConfigurationUtil.setInputClass(configuration,inputClass);
        return configuration;
    }

    /**
     * A method to override which HBase cluster for {@link PhoenixInputFormat} to read from
     * @param job MapReduce Job
     * @param quorum an HBase cluster's ZooKeeper quorum
     */
    public static void setInputCluster(final Job job, final String quorum) {
        final Configuration configuration = job.getConfiguration();
        PhoenixConfigurationUtil.setInputCluster(configuration, quorum);
    }
    /**
     *
     * @param job
     * @param outputClass
     * @param tableName  Output table
     * @param columns    List of columns separated by ,
     */
    public static void setOutput(final Job job, final String tableName,final String columns) {
        job.setOutputFormatClass(PhoenixOutputFormat.class);
        final Configuration configuration = job.getConfiguration();
        PhoenixConfigurationUtil.setOutputTableName(configuration, tableName);
        PhoenixConfigurationUtil.setUpsertColumnNames(configuration,columns.split(","));
    }


    /**
     *
     * @param job
     * @param outputClass
     * @param tableName  Output table
     * @param fieldNames fields
     */
    public static void setOutput(final Job job, final String tableName , final String... fieldNames) {
          job.setOutputFormatClass(PhoenixOutputFormat.class);
          final Configuration configuration = job.getConfiguration();
          PhoenixConfigurationUtil.setOutputTableName(configuration, tableName);
          PhoenixConfigurationUtil.setUpsertColumnNames(configuration,fieldNames);
    }

    /**
     * A method to override which HBase cluster for {@link PhoenixOutputFormat} to write to
     * @param job MapReduce Job
     * @param quorum an HBase cluster's ZooKeeper quorum
     */
    public static void setOutputCluster(final Job job, final String quorum) {
        final Configuration configuration = job.getConfiguration();
        PhoenixConfigurationUtil.setOutputCluster(configuration, quorum);
    }

    public static TableTunnel.DownloadSession getDownloadSession(Configuration conf)
            throws TunnelException {
        String accessKeyId = conf.get(ODPSMapper.ACCESS_KEY_ID_CONFKEY);
        String accessKeySecret = conf.get(ODPSMapper.ACCESS_KEY_SECRET_CONFKEY);
        String project = conf.get(ODPSMapper.ODPS_PROJECT_CONFKEY);
        String odpsUrl = conf.get(ODPSMapper.ODPS_URL_CONFKEY);
        String tunnelUrl = conf.get(ODPSMapper.ODPS_TUNNEL_URL_CONFKEY);
        String tableName = conf.get(ODPSMapper.ODPS_TABLE_NAME_CONFKEY);
        String partitionSpec = conf.get(ODPSMapper.ODPS_TABLE_PARTITION_SPEC_CONFKEY, "Non-Partitioned");

        AliyunAccount account = new AliyunAccount(accessKeyId, accessKeySecret);
        Odps odps = new Odps(account);
        odps.setDefaultProject(project);
        odps.setEndpoint(odpsUrl);
        TableTunnel tunnel = new TableTunnel(odps);
        tunnel.setEndpoint(tunnelUrl);

        if (partitionSpec.equals("Non-Partitioned")) {
            return tunnel.createDownloadSession(project, tableName);
        } else {
            PartitionSpec parSpec = new PartitionSpec(partitionSpec);
            return tunnel.createDownloadSession(project, tableName, parSpec);
        }
    }

    public static List<ColumnInfo> buildTargetTableColumns(Configuration conf) {
        return Lists.newArrayList(
                Iterables.transform(
                        Splitter.on("|").split(conf.get(ODPSMapper.PHOENIX_MAPPED_COLUMN_INFO_CONFKEY)),
                        new Function<String, ColumnInfo>() {
                            @Nullable
                            @Override
                            public ColumnInfo apply(@Nullable String input) {
                                if (input == null || input.isEmpty()) {
                                    return null;
                                }
                                return ColumnInfo.fromString(input);
                            }
                        }));
    }

    public static List<Pair<Long, Long>> getRanges(long count, int numRanges) {
        List<Pair<Long, Long>> ranges = new ArrayList<>();
        if (count <= numRanges) {
            ranges.add(new Pair<>(0L, count));
        } else {
            long[] rangeSpans = new long[numRanges];
            long per = count / numRanges;
            Arrays.fill(rangeSpans, per);

            long remainder = count - (numRanges * per) + 1;
            for (int i = 0; i < remainder; i++) {
                rangeSpans[i] += 1;
            }

            long start = 0, end = 0;
            for (int i = 0; i < numRanges && end < count; i++) {
                end = start + rangeSpans[i] - 1;
                end = end < count ? end : count;
                ranges.add(new Pair<>(start, end));
                start = start < count ? end + 1 : count;
            }
        }
        return ranges;
    }

    /*
    Map all unique pairs <family, name>  to index. Table name is part of TableRowkey, so we do
    not care about it
     */
    public static Map<byte[], Integer> initColumnIndexes(PhoenixConnection conn, List<String> logicalNames) throws
            SQLException {
        Map<byte[], Integer> columnIndexes = new TreeMap<>(Bytes.BYTES_COMPARATOR);
        int columnIndex = 0;
        for (int index = 0; index < logicalNames.size(); index++) {
            PTable table = PhoenixRuntime.getTable(conn, logicalNames.get(index));
            if (!table.getImmutableStorageScheme().equals(PTable.ImmutableStorageScheme.ONE_CELL_PER_COLUMN)) {
                List<PColumnFamily> cfs = table.getColumnFamilies();
                for (int i = 0; i < cfs.size(); i++) {
                    byte[] family = cfs.get(i).getName().getBytes();
                    byte[] cfn = Bytes.add(family, QueryConstants.NAMESPACE_SEPARATOR_BYTES,
                            QueryConstants.SINGLE_KEYVALUE_COLUMN_QUALIFIER_BYTES);
                    columnIndexes.put(cfn, new Integer(columnIndex));
                    columnIndex++;
                }
            } else {
                List<PColumn> cls = table.getColumns();
                for (int i = 0; i < cls.size(); i++) {
                    PColumn c = cls.get(i);
                    byte[] family = new byte[0];
                    byte[] cq;
                    if (!SchemaUtil.isPKColumn(c)) {
                        family = c.getFamilyName().getBytes();
                        cq = c.getColumnQualifierBytes();
                    } else {
                        cq = c.getName().getBytes();
                    }
                    byte[] cfn = Bytes.add(family, QueryConstants.NAMESPACE_SEPARATOR_BYTES, cq);
                    if (!columnIndexes.containsKey(cfn)) {
                        columnIndexes.put(cfn, new Integer(columnIndex));
                        columnIndex++;
                    }
                }
                byte[] emptyColumnFamily = SchemaUtil.getEmptyColumnFamily(table);
                byte[] emptyKeyValue = EncodedColumnsUtil.getEmptyKeyValueInfo(table).getFirst();
                byte[] cfn = Bytes.add(emptyColumnFamily, QueryConstants.NAMESPACE_SEPARATOR_BYTES, emptyKeyValue);
                columnIndexes.put(cfn, new Integer(columnIndex));
                columnIndex++;
            }
        }
        return columnIndexes;
    }

    /**
     * Find the column index which will replace the column name in
     * the aggregated array and will be restored in Reducer
     *
     * @param cell       KeyValue for the column
     * @return column index for the specified cell or -1 if was not found
     */
    private static int findIndex(Cell cell, Map<byte[], Integer> columnIndexes) throws IOException {
        byte[] familyName = Bytes.copy(cell.getFamilyArray(), cell.getFamilyOffset(),
                cell.getFamilyLength());
        byte[] cq = Bytes.copy(cell.getQualifierArray(), cell.getQualifierOffset(),
                cell.getQualifierLength());
        byte[] cfn = Bytes.add(familyName, QueryConstants.NAMESPACE_SEPARATOR_BYTES, cq);
        if(columnIndexes.containsKey(cfn)) {
            return columnIndexes.get(cfn);
        }
        return -1;
    }

    /**
     * Collect all column values for the same Row. RowKey may be different if indexes are involved,
     * so it writes a separate record for each unique RowKey
     *
     * @param context    Current mapper context
     * @param tableName Table index in tableNames list
     * @param lkv        List of KV values that will be combined in a single ImmutableBytesWritable
     * @throws IOException
     * @throws InterruptedException
     */

    public static void writeAggregatedRow(Mapper.Context context, String tableName, List<KeyValue> lkv, Map<byte[], Integer> columnIndexes)
            throws IOException, InterruptedException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream(1024);
        DataOutputStream outputStream = new DataOutputStream(bos);
        ImmutableBytesWritable outputKey =null;
        if (!lkv.isEmpty()) {
            for (KeyValue cell : lkv) {
                if (outputKey == null || Bytes.compareTo(outputKey.get(), outputKey.getOffset(),
                        outputKey.getLength(), cell.getRowArray(), cell.getRowOffset(), cell
                                .getRowLength()) != 0) {
                    // This a the first RowKey or a different from previous
                    if (outputKey != null) { //It's a different RowKey, so we need to write it
                        ImmutableBytesWritable aggregatedArray =
                                new ImmutableBytesWritable(bos.toByteArray());
                        outputStream.close();
                        context.write(new TableRowkeyPair(tableName, outputKey), aggregatedArray);
                    }
                    outputKey = new ImmutableBytesWritable(cell.getRowArray(), cell.getRowOffset()
                            , cell.getRowLength());
                    bos = new ByteArrayOutputStream(1024);
                    outputStream = new DataOutputStream(bos);
                }
                /*
                The order of aggregation: type, index of column, length of value, value itself
                 */
                int i = findIndex(cell, columnIndexes);
                if(i == -1) {
                    //That may happen when we load only local indexes. Since KV pairs for both
                    // table and local index are going to the same physical table at that point
                    // we skip those KVs that are not belongs to loca index
                    continue;
                }
                outputStream.writeByte(cell.getTypeByte());
                WritableUtils.writeVLong(outputStream,cell.getTimestamp());
                WritableUtils.writeVInt(outputStream, i);
                WritableUtils.writeVInt(outputStream, cell.getValueLength());
                outputStream.write(cell.getValueArray(), cell.getValueOffset(), cell.getValueLength());

            }
            ImmutableBytesWritable aggregatedArray = new ImmutableBytesWritable(bos.toByteArray());
            outputStream.close();
            context.write(new TableRowkeyPair(tableName, outputKey), aggregatedArray);
        }
    }
}
