package com.qiho.manager.biz.runnable;

import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import com.google.common.base.Charsets;
import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.io.CharSource;
import com.google.common.io.Resources;
import com.qiho.manager.biz.vo.component.ComponentCodeVO;
import com.qiho.manager.common.constant.CacheConstantseEnum;
import com.qiho.manager.common.exception.QihoManagerException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.net.URL;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;

/**
 * 下载组件代码runnable
 *
 * @author chensong
 * @create 2018-07-31
 */
@Component
@Scope("prototype")
public class DownloadComponentCodeRunnable implements Runnable {

    private static final Logger LOGGER = LoggerFactory.getLogger(DownloadComponentCodeRunnable.class);

    /** 代码之间的分隔符 */
    private static final String CODE_SPLIT = "<split>";

    private static final Joiner LINE_JOINER =  Joiner.on("");

    private static final Splitter CODE_SPLITTER = Splitter.on(CODE_SPLIT);

    private static final int DEAULT_CODE_NUM = 3;

    @Resource(name = "redisTemplate")
    private AdvancedCacheClient advancedCacheClient;

    /** 文件地址 */
    private String fileUrl;

    /** 组件id */
    private Long componentId;

    /** 计数器 */
    private CountDownLatch latch;

    /** 存储结果 */
    private Map<Long, ComponentCodeVO> resultMap;

    @Override
    public void run() {

        try {
            if (StringUtils.isBlank(fileUrl)) {
                throw new QihoManagerException("文件地址为空");
            }

            // 从缓存获取代码
            String code = getCodeFormCache();
            if (StringUtils.isNotBlank(code)) {
                // 从缓存中获取到代码 将代码拆分
                splitCodeTOMap(code);

            } else {
                // 未从缓存中获取到代码 则从Oss获取代码
                URL url = new URL(fileUrl);
                CharSource charset = Resources.asCharSource(url, Charsets.UTF_8);
                List<String> lines = charset.readLines();
                code = LINE_JOINER.join(lines);

                if (StringUtils.isBlank(code)) {
                    LOGGER.error("oss读取组件代码为空，组件id=[{}]，fileUrl=[{}] ", componentId, fileUrl);
                    throw new QihoManagerException("代码获取为空");
                }

                // 将代码拆分
                splitCodeTOMap(code);

                // 将组件代码set Redis
                setCodeOnRedis(code);
            }

        } catch (Exception e){
            LOGGER.error("组件代码解析错误， componenntId=[{}], fileUrl=[{}]", componentId, fileUrl, e);

        } finally {
            latch.countDown();
        }
    }


    /**
     * 将代码拆分并把拆分后的代码放进结果集合
     * @param code

     */
    private void splitCodeTOMap(String code){
        List<String> codeList = CODE_SPLITTER.splitToList(code);
        if (codeList.size() != DEAULT_CODE_NUM) {
            LOGGER.error("组件代码拆分异常，组件id=[{}]，fileUrl=[{}]", componentId, fileUrl);
            throw new QihoManagerException("代码拆分异常");
        }

        ComponentCodeVO componentCode = new ComponentCodeVO();
        componentCode.setHtmlCode(codeList.get(0));
        componentCode.setCssCode(codeList.get(1));
        componentCode.setJsCode(codeList.get(2));

        resultMap.put(componentId, componentCode);
    }

    /**
     * 从缓存中获取组件代码
     * @return
     */
    private String getCodeFormCache(){
        String key = CacheConstantseEnum.COMPONENT_FILE_URL.toString() + ":" + componentId;
        return  advancedCacheClient.get(key);
    }

    /**
     * 将组件代码set 进缓存
     * @param code
     */
    private void setCodeOnRedis(String code){
        String key = CacheConstantseEnum.COMPONENT_FILE_URL.toString() + ":" + componentId;
        advancedCacheClient.set(key, code, CacheConstantseEnum.COMPONENT_FILE_URL.getTime(),
                CacheConstantseEnum.COMPONENT_FILE_URL.getTimeUnit());
    }

    public void setComponentId(Long componentId) {
        this.componentId = componentId;
    }

    public void setFileUrl(String fileUrl) {
        this.fileUrl = fileUrl;
    }

    public void setLatch(CountDownLatch latch) {
        this.latch = latch;
    }

    public void setResultMap(Map<Long, ComponentCodeVO> resultMap) {
        this.resultMap = resultMap;
    }
}
