package com.qiho.manager.biz.service;


import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Objects;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.params.ErpOrderPageParams;
import com.qiho.manager.biz.runnable.AbstractOrderHandler;
import com.qiho.manager.biz.runnable.WdtErpOrderExportRunnable;
import com.qiho.manager.biz.vo.OrderDealWithVO;
import com.qiho.manager.biz.vo.OrderTaskInfoVO;
import com.qiho.manager.common.constant.CacheConstantseEnum;
import com.qiho.manager.common.util.OSSFileService;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.LineIterator;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * Created by liuyao on 2017/6/8.
 */
@Service
public class OrderDealWithService implements ApplicationContextAware {

    private Logger logger = LoggerFactory.getLogger(OrderDealWithService.class);

    private static final int BATCHE_NUM = 1000;

    private CacheConstantseEnum constantse = CacheConstantseEnum.DEAL_WITH_ORDER;

    private ApplicationContext applicationContext;
    @Autowired
    private OSSFileService ossFileService;
    @Resource(name = "stringRedisTemplate")
    private StringRedisTemplate redisTemplate;

    @Resource
    private ExecutorService executorService;
    /**
     * 提交
     * @param fileUrl 文件url
     * @param status 要改变的状态
     * @param handlerClass
     */
    public OrderDealWithVO submitDealWithTask(String fileUrl, String status, Class<? extends AbstractOrderHandler> handlerClass,JSONObject extraParams){
        OrderDealWithVO vo = new OrderDealWithVO();
        InputStream input = ossFileService.getOssFileInputStream(fileUrl);
        //加载文件
        try {
            if(Objects.equal(null,input)){
                throw new QihoException("文件不存在");
            }
            String taskId = createTaskId();
            int count = 0;
            String cacheKey = constantse.getCacheKey(taskId);
            List<String> lines = Lists.newArrayList();
            LineIterator iterator = IOUtils.lineIterator(input,Charset.forName("GBK"));
            AbstractOrderHandler handler = applicationContext.getBean(handlerClass);
            while (iterator.hasNext()){
                String line = StringUtils.trim(iterator.next());
                if(StringUtils.isNotBlank(line)){
                    lines.add(line);
                }
                if(lines.size()>=BATCHE_NUM){
                    count += flush(cacheKey,lines,handler,extraParams,status);
                }
            }
            count += flush(cacheKey,lines,handler,extraParams,status);
            // taskCount ?任务批次
            redisTemplate.boundHashOps(cacheKey).increment("taskCount",count%BATCHE_NUM==0?count/BATCHE_NUM:(count/BATCHE_NUM+1));
            redisTemplate.boundHashOps(cacheKey).put("count",String.valueOf(count));
            if(count==0){
                redisTemplate.boundHashOps(cacheKey).put("success","true");
            }
            redisTemplate.expire(cacheKey,3, TimeUnit.HOURS);//3小时后删除在缓存中的任务元数据
            vo.setTaskId(taskId);
            vo.setCount(count);
        }catch(Exception e){
            logger.error("订单任务失败",e);
        }finally {
            ossFileService.closeInputStream(input);
        }
        return vo;
    }

    /**
     * 提交
     * @param fileUrl
     * @param handlerClass
     */
    public OrderDealWithVO submitDealWithTask(String fileUrl, Class<? extends AbstractOrderHandler> handlerClass){
        return submitDealWithTask(fileUrl,null,handlerClass,null);
    }
    
    /**
     * 提交
     * @param fileUrl
     * @param status
     * @param handlerClass
     */
    public OrderDealWithVO submitDealWithTask(String fileUrl, String status, Class<? extends AbstractOrderHandler> handlerClass){
        return submitDealWithTask(fileUrl,status,handlerClass,null);
    }
    
    /**
     * 提交
     * @param fileUrl
     * @param extraParams
     * @param handlerClass
     */
    public OrderDealWithVO submitDealWithTask(String fileUrl, Class<? extends AbstractOrderHandler> handlerClass,JSONObject extraParams){
        return submitDealWithTask(fileUrl,null,handlerClass,extraParams);
    }

    private <T> Integer flush(String cacheKey, List<String> lines, AbstractOrderHandler<T> handler,JSONObject extraParams, String status){
        if(lines.isEmpty()){
            return 0;
        }

        List<T> list = Lists.newArrayList();
        for(String line:lines){
            T params = null;
            if(StringUtils.isBlank(status)){
                params = handler.transform(line);
            }else{
                params = handler.transform(line,status);
            }
            if(params!=null){
                list.add(params);
            }
        }
        int total = 0;
        if(list.isEmpty()){
            return total;
        }
        if(Objects.equal(null,extraParams) || extraParams.isEmpty()){
            total = handler.exeTask(cacheKey,list);
        }else{
            total = handler.exeTask(cacheKey,list,extraParams);
        }
        lines.clear();
        return total;
    }

    /**
     * 查询订单处理状态
     * @param taskId
     * @return
     */
    public JSONObject findDealWithInfo(String taskId){
        BoundHashOperations<String,String,Object> operations = redisTemplate.boundHashOps(taskId);
        Map<String,Object> info = operations.entries();
        JSONObject json = new JSONObject();
        if(!info.isEmpty()){
            json.putAll(info);
        }
        json.put("success",json.getBoolean("success"));
        return json;
    }

    /**
     * 查询订单处理任务状态
     * @param taskId
     * @return
     */
    public OrderTaskInfoVO findOrderTaskInfo(String taskId){
        BoundHashOperations<String,String,String> operations = redisTemplate.boundHashOps(taskId);
        Map<String,String> info = operations.entries();
        if(info.isEmpty()){
            return null;
        }
        OrderTaskInfoVO vo = new OrderTaskInfoVO();
        int handleCount = Integer.valueOf(info.get("handleCount"));
        int totalCount = Integer.valueOf(info.get("totalCount"));
        String status = info.get("status");
        String failLines = info.get("failLines");
        vo.setHandleCount(handleCount);
        vo.setTotalCount(totalCount);
        vo.setFailLines(failLines);
        vo.setStatus(status);
        if(status.equals("over")){
            vo.setSuccess(true);
            redisTemplate.delete(taskId);
        }else{
            vo.setSuccess(false);
        }
        return vo;
    }


    /**
     * 生成任务Id
     * @return
     */
    public String createTaskId(){
        Long taskId = new Date().getTime();
        String key = constantse.getCacheKey(taskId);
        Map<String,String> info = Maps.newHashMap();
        info.put("success","false");
        info.put("count","0");
        info.put("successCount","0");
        info.put("failCount","0");
        info.put("taskCount","0");
        //失败的数据id,erp数据处理是erpId逗号分隔,用于标识处理csv数据时哪一条数据处理失败了
        info.put("failIds","");
        redisTemplate.opsForHash().putAll(key,info);
        //3小时后删除在缓存中的任务元数据
        redisTemplate.expire(key,3, TimeUnit.HOURS);
        return key;
    }



    @Override
    public void setApplicationContext(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
    }


    public Map<String,Object> submitOrdersExport(ErpOrderPageParams params) {
        WdtErpOrderExportRunnable runnable = applicationContext.getBean(WdtErpOrderExportRunnable.class);
        runnable.setParams(params);
        Map<String,Object> info = runnable.init();
        executorService.submit(runnable);
        return info;
    }

}
