package com.qiho.manager.biz.service.merchant.impl;

import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.google.common.io.CharSink;
import com.google.common.io.FileWriteMode;
import com.google.common.io.Files;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.merchant.MerchantUndeliveryDto;
import com.qiho.center.api.params.merchant.MerchantUndeliveryQueryParams;
import com.qiho.center.api.remoteservice.merchant.RemoteMerchantUndeliveryBackendService;
import com.qiho.manager.biz.service.merchant.MerchantUndeliveryService;
import com.qiho.manager.biz.vo.Pagenation;
import com.qiho.manager.biz.vo.merchant.MerchantUndeliveryVO;
import com.qiho.manager.biz.vo.merchant.UndeliveryExportVO;
import com.qiho.manager.common.exception.QihoManagerException;
import com.qiho.manager.common.util.UploadTool;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.IteratorUtils;
import org.apache.commons.collections.ListUtils;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;
import org.joda.time.LocalDateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * 商家不配送service
 *
 * @author peanut.huang
 * @date 2018/1/10.
 */
@Service
public class MerchantUndeliveryServiceImpl implements MerchantUndeliveryService {

    private static final Logger LOGGER = LoggerFactory.getLogger(MerchantUndeliveryServiceImpl.class);

    @Resource
    private RemoteMerchantUndeliveryBackendService remoteMerchantUndeliveryBackendService;

    @Override
    public Pagenation<MerchantUndeliveryVO> find4Page(Long merchantId, String regionCode, Integer pageNum, Integer pageSize) {

        Pagenation<MerchantUndeliveryVO> result = new Pagenation<>();

        //查询参数
        MerchantUndeliveryQueryParams queryParams = new MerchantUndeliveryQueryParams();
        queryParams.setMerchantId(merchantId);
        queryParams.setRegionCode(regionCode);
        queryParams.setMax(pageSize);
        queryParams.setOffset((pageNum - 1) * pageSize);

        //查询
        PagenationDto<MerchantUndeliveryDto> pagenationDto;
        try {
            pagenationDto = remoteMerchantUndeliveryBackendService.find4Page(queryParams);
        }catch (Exception e){
            LOGGER.error("find page failed", e);
            return result;
        }

        result.setList(BeanUtils.copyList(pagenationDto.getList(), MerchantUndeliveryVO.class));
        result.setTotal(pagenationDto.getTotal());

        return result;
    }

    @Override
    public boolean deleteUndelivery(List<Long> idList) {
        if(CollectionUtils.isEmpty(idList)){
            return false;
        }

        try {
            return remoteMerchantUndeliveryBackendService.deleteByIds(idList);
        }catch (Exception e){
            LOGGER.error("delete undelivery failed", e);
            return false;
        }
    }

    @Override
    public void importUndelivery(String encodeType, MultipartFile file) {

        // parse file to csvParser
        CSVParser csvParser = parseFile(encodeType, file);

        // check column header
        checkHeader(csvParser.getHeaderMap());

        // build batch insert dto list
        List<MerchantUndeliveryDto> undeliveryDtoList = buildUndeliveryDtoList(csvParser);
        if(CollectionUtils.isEmpty(undeliveryDtoList)){
            throw new QihoManagerException("导入实体为空");
        }

        // invoke remote method
        Boolean result;
        try {
            result = remoteMerchantUndeliveryBackendService.batchInsert(undeliveryDtoList);
        }catch (Exception e){
            LOGGER.error("remote invoke batch insert failed", e);
            throw new QihoManagerException("导入失败，请管理员查看日志");
        }

        if(!result){
            throw new QihoManagerException("导入不成功或数据都已存在");
        }
    }


    private static final String EXPORT_PATH = "merchant/undelivery/";
    private static final Joiner joiner = Joiner.on(",");
    @Override
    public UndeliveryExportVO exportUndelivery(Long merchantId, String regionCode, String encodeType, String fileName) {

        UndeliveryExportVO result = new UndeliveryExportVO();
        try {
            File file = new File(Files.createTempDir(), fileName + ".csv");
            CharSink charSink = Files.asCharSink(file, Charset.forName(encodeType), FileWriteMode.APPEND);

            //set header
            charSink.writeLines(Lists.newArrayList(joiner.join(EXPORT_HEADERS)));

            // fetch export data
            List<MerchantUndeliveryDto> remoteDtoList = fetchExportData(merchantId, regionCode);

            if(CollectionUtils.isNotEmpty(remoteDtoList)){
                //set size
                result.setCount(remoteDtoList.size());

                // set records
                for (MerchantUndeliveryDto dto : remoteDtoList) {

                    String [] record = { dto.getMerchantName(),
                                         dto.getMerchantId().toString(),
                                         dto.getRegionCode(),
                                         LocalDateTime.fromDateFields(dto.getGmtCreate()).toString("yyyy-MM-dd HH:mm:ss")
                                        };

                    charSink.writeLines(Lists.newArrayList(joiner.join(record)));
                }
            }

            // upload
            String path = UploadTool.uploadOssNotCDN(file, EXPORT_PATH + fileName + ".csv", "application/csv;charset="+ encodeType);

            result.setFilePath(path);

        }catch (Exception e){
            LOGGER.error("", e);
            throw new QihoManagerException("导出处理异常，请查看日志");
        }

        return result;
    }

    /**
     * fetch export data
     *
     * @param merchantId
     * @param regionCode
     * @return
     */
    private List<MerchantUndeliveryDto> fetchExportData(Long merchantId, String regionCode) {
        //查询参数
        MerchantUndeliveryQueryParams queryParams = new MerchantUndeliveryQueryParams();
        queryParams.setMerchantId(merchantId);
        queryParams.setRegionCode(regionCode);
        queryParams.setOffset(null);
        queryParams.setMax(null);

        try {
            return  remoteMerchantUndeliveryBackendService.findByParams(queryParams);
        }catch (Exception e){
            LOGGER.error("find by params failed", e);
            return Collections.emptyList();
        }
    }

    /**
     * build dto list
     *
     * @param csvParser
     * @return
     */
    private List<MerchantUndeliveryDto> buildUndeliveryDtoList(CSVParser csvParser) {

        List<CSVRecord> recordList;
        try {
            recordList = csvParser.getRecords();
        } catch (IOException e) {
            LOGGER.error("csvParser get records failed", e);
            return Collections.emptyList();
        }

        List<MerchantUndeliveryDto> result = Lists.newArrayListWithExpectedSize(recordList.size());

        recordList.forEach(csvRecord -> {

            MerchantUndeliveryDto dto = new MerchantUndeliveryDto();
            List record = IteratorUtils.toList(csvRecord.iterator());

            if(CollectionUtils.isNotEmpty(record) && record.size() >= 3){

                //first column is merchantName
                String merchantName = record.get(0).toString();
                //second column is merchantId
                Long merchantId = Long.valueOf(record.get(1).toString());
                //third column is regionCode
                String regionCode = record.get(2).toString();

                dto.setMerchantName(merchantName);
                dto.setMerchantId(merchantId);
                dto.setRegionCode(regionCode);

                result.add(dto);
            }
        });

        return result;
    }


    private static final List<String> UNDELIVERY_HEADERS = Arrays.asList("商家名称","商家id","行政区域代码");

    private static final String [] EXPORT_HEADERS = {"商家名称","商家id","行政区域代码","添加时间"};

    /**
     * check header
     *
     * @param headerMap key is column name, values is column index
     */
    private void checkHeader(Map<String, Integer> headerMap) {

        List headerList = Lists.newArrayList(headerMap.keySet());

        if(!ListUtils.isEqualList(headerList, UNDELIVERY_HEADERS)){
            throw new QihoManagerException("列头不正确");
        }
    }

    /**
     * parse file
     *
     * @param encodeType
     * @param file
     * @return
     */
    private CSVParser parseFile(String encodeType, MultipartFile file) {
        try {
            // file to inputStream by encodeType
            InputStreamReader isr = new InputStreamReader(file.getInputStream(),encodeType);

            // to csv parser
            return CSVFormat.EXCEL.withIgnoreEmptyLines().withHeader().parse(isr);
        } catch (IOException e) {
            LOGGER.error("parse file to record failed", e);
            throw new QihoManagerException("文件解析失败");
        }
    }
}
