package com.qiho.manager.biz.runnable.itemtaskhandler;

import cn.com.duiba.sso.api.domain.dto.AdminDto;
import cn.com.duiba.sso.api.tool.RequestTool;
import com.google.common.base.Splitter;
import com.qiho.center.api.dto.region.BaiqiRegionDto;
import com.qiho.center.api.enums.item.ItemEvaluateTypeEnum;
import com.qiho.center.api.params.item.BatchImportItemEvaluateParams;
import com.qiho.center.api.remoteservice.item.RemoteItemEvaluateBackendService;
import com.qiho.manager.biz.runnable.AbstractItemEvaluateHandler;
import com.qiho.manager.biz.service.region.BaiqiRegionService;
import com.qiho.manager.biz.vo.region.RegionVO;
import com.qiho.manager.common.util.MobileUtil;
import com.qiho.manager.common.util.StringUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 批量导入商品评价
 *
 * @author jiafeng.geng
 * @date 2018-8-29
 */
@Service
public class ItemEvaluateHandler implements AbstractItemEvaluateHandler<BatchImportItemEvaluateParams> {

    private static final Logger LOGGER = LoggerFactory.getLogger(ItemEvaluateHandler.class);

    private Splitter splitter = Splitter.on(",").trimResults().omitEmptyStrings();

    @Resource
    private RemoteItemEvaluateBackendService remoteItemEvaluateBackendService;

    @Resource
    private BaiqiRegionService baiqiRegionService;

    @Override
    public BatchImportItemEvaluateParams transform(String row, Long itemId) {
        List<String> line = splitter.splitToList(row);
        return line.size() < 6 ? null : getBatchImportItemEvaluateParams(line, itemId);
    }

    @Override
    public Integer exeTask(String cacheKey, List<BatchImportItemEvaluateParams> list) {
        return remoteItemEvaluateBackendService.batchImportItemEval(cacheKey, list);
    }

    private BatchImportItemEvaluateParams getBatchImportItemEvaluateParams(List<String> line, Long itemId) {

        BatchImportItemEvaluateParams params = new BatchImportItemEvaluateParams();

        // 获取当前操作人信息
        try {
            AdminDto admin = RequestTool.getAdmin();
            params.setOperator(admin != null ? admin.getAccount() : StringUtils.EMPTY);
        } catch (Exception e) {
            LOGGER.warn("获取管理员异常, params: {} ", params, e);
            params.setOperator(StringUtils.EMPTY);
        }

        try {

            // 获取当前最大排序值
            Integer maxSort = remoteItemEvaluateBackendService.queryMaxSort(itemId);
            params.setSort(maxSort == null ? 0 : maxSort + 1);

            // 判断是否为直辖市、自治区
            RegionVO regionVO = baiqiRegionService.findByNameAndLevel(line.get(3), 1);
            if (null != regionVO && regionVO.getRegionName().endsWith("市")) {
                params.setRegionName(subRegionSuffix(line.get(3)));
            } else {
                params.setRegionName(subRegionSuffix(line.get(3)) + subRegionSuffix(line.get(4)));
            }

            // 捕获星级评分解析异常
            params.setItemId(itemId);
            params.setUserName(line.get(0));
            params.setMobile(line.get(1));
            params.setRating(parseRating(line.get(2)));
            params.setProvince(subRegionSuffix(line.get(3)));
            params.setCity(subRegionSuffix(line.get(4)));
            params.setEvaluateContent(line.get(5));
            params.setEvaluateType(ItemEvaluateTypeEnum.EXCEL_IMPORT_EVAL.getCode());

        } catch (Exception e) {
            LOGGER.info("解析Excel内容出错, line={}", line, e);
            return null;
        }

        return checkParam(params) ? params : null;
    }

    private int parseRating(String num) {
        return StringUtils.isNumeric(num) ? Integer.parseInt(num) : 0;
    }

    /**
     * 截取省市的后缀
     *
     * @param regionStr
     * @return
     */
    private String subRegionSuffix(String regionStr) {
        if (regionStr.endsWith("省") || regionStr.endsWith("市")) {
            return regionStr.substring(0, regionStr.length() - 1);
        }
        if (regionStr.endsWith("自治区")) {
            return regionStr.substring(0, 2);
        }
        return regionStr;
    }

    /**
     * 参数校验
     *
     * @param param
     * @return
     */
    private boolean checkParam(BatchImportItemEvaluateParams param) {
        if (null == param) {
            return false;
        }

        // 校验姓名是否为纯中文
        String userName = param.getUserName();
        if (!StringUtil.isFullChinese(userName)) {
            return false;
        }

        // 校验姓名长度
        if (userName.length() < 2 || userName.length() > 10) {
            return false;
        }

        // 校验手机号
        if (!MobileUtil.validPhoneNum(param.getMobile())) {
            return false;
        }

        // 校验星级评分
        Integer rating = param.getRating();
        if (null == rating || rating <= 0 || rating > 5) {
            return false;
        }

        // 校验省
        RegionVO provinceRegionVO = baiqiRegionService.findByNameAndLevel(param.getProvince(), 1);
        if (null == provinceRegionVO) {
            return false;
        }

        // 校验市, 不校验直辖市
        if (!provinceRegionVO.getRegionName().endsWith("市") && StringUtils.isNotEmpty(param.getCity())) {

            RegionVO cityRegionVO = baiqiRegionService.findByNameAndLevel(param.getCity(), 2);
            if (null == cityRegionVO) {
                return false;
            }

            // 当前省是否包含当前市
            List<BaiqiRegionDto> regionDtoList = baiqiRegionService.find(provinceRegionVO.getId());
            List<String> regionStrList = regionDtoList.stream().map(BaiqiRegionDto::getRegionName)
                    .collect(Collectors.toList());

            if (!regionStrList.contains(cityRegionVO.getRegionName())) {
                return false;
            }

        }

        // 校验评论内容长度
        if (param.getEvaluateContent().length() > 50) {
            return false;
        }

        return true;
    }
}
