package com.qiho.manager.biz.service.logistics.impl;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.logistics.LogisticsDto;
import com.qiho.center.api.dto.logistics.LogisticsNicknameDto;
import com.qiho.center.api.params.logitics.LogisticsQueryParam;
import com.qiho.center.api.remoteservice.logistics.RemoteLogisticsService;
import com.qiho.manager.biz.service.logistics.LogisticsService;
import com.qiho.manager.biz.vo.Pagenation;
import com.qiho.manager.biz.vo.logistics.LogisticsNicknameVO;
import com.qiho.manager.biz.vo.logistics.LogisticsVO;
import com.qiho.manager.common.exception.QihoManagerException;
import org.apache.commons.collections.MapUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 物流公司服务
 * Author: zhanghuifeng
 * Date:   10:31 2017/12/28
 */
@Service("logisticsService")
public class LogisticsServiceImpl implements LogisticsService {

    @Resource
    private RemoteLogisticsService remoteLogisticsService;

    //存放所有百奇物流公司缓存
    private static final String BAIQILOGISTICSLOCALMAPCACHEKEY = "baiqiLogisticsLocalMapCacheKey";
    private static final Cache<String,Map<String, String>> baiqiLogisticsLocalMapCache = CacheBuilder.newBuilder().maximumSize(5).expireAfterWrite(5, TimeUnit.MINUTES).build();

    @Override
    public List<LogisticsDto> findAll() {
        return remoteLogisticsService.findAll();
    }

    @Override
    public Pagenation<LogisticsVO> querypage(LogisticsQueryParam param) {
        PagenationDto<LogisticsDto> pagenationDto = remoteLogisticsService.queryPage(param);
        Pagenation<LogisticsVO> page = new Pagenation<>();
        page.setTotal(pagenationDto.getTotal());
        page.setList(BeanUtils.copyList(pagenationDto.getList(), LogisticsVO.class));
        return page;
    }

    @Override
    public List<LogisticsNicknameVO> findNickname(Long logisticsId) {
        return BeanUtils.copyList(remoteLogisticsService.findNickname(logisticsId), LogisticsNicknameVO.class);
    }

    @Override
    public Boolean addNickname(Long logisticsId, String nickname) {
        LogisticsNicknameDto dto = new LogisticsNicknameDto();
        dto.setLogisticsId(logisticsId);
        dto.setNickname(nickname);
        try {
            return remoteLogisticsService.addNickname(dto);
        } catch (BizException e) {
            throw new QihoManagerException(e.getMessage());
        }
    }

    @Override
    public Boolean deleteNickname(Long id) {
        try {
            return remoteLogisticsService.deleteNickname(id);
        } catch (BizException e) {
            throw new QihoManagerException(e.getMessage());
        }
    }

    @Override
    public String insertLogistics(String logisticsCode, String logisticsName) {
        LogisticsDto logisticsDto = new LogisticsDto();
        logisticsDto.setLogisticsCode(logisticsCode);
        logisticsDto.setLogisticsName(logisticsName);
        Boolean result = remoteLogisticsService.insertLogistics(logisticsDto);
        return new StringBuilder("物流公司【")
                .append(logisticsName)
                .append("】插入")
                .append(result ? "成功" : "失败")
                .toString();
    }

    @Override
    public String deleteLogistics(Long id) {
        Boolean result = remoteLogisticsService.deleteLogistics(id);
        return new StringBuilder("物流公司id=")
                .append(id)
                .append("删除")
                .append(result ? "成功" : "失败")
                .toString();
    }

    @Override
    public Map<String, String> getBaiqiLogisticsCodeToName() {
        Map<String, String> map = baiqiLogisticsLocalMapCache.getIfPresent(BAIQILOGISTICSLOCALMAPCACHEKEY);
        if (MapUtils.isEmpty(map)){
            List<LogisticsDto> logisticsDtos = remoteLogisticsService.findAll();
            map = logisticsDtos.stream().collect(Collectors.toMap(LogisticsDto::getLogisticsCode, LogisticsDto::getLogisticsName, (a, b) -> b));
            baiqiLogisticsLocalMapCache.put(BAIQILOGISTICSLOCALMAPCACHEKEY,map);
        }
        return map;
    }
}
