package com.qiho.manager.biz.service.page.impl;

import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.collect.Lists;
import com.qiho.center.api.dto.page.CollItemDto;
import com.qiho.center.api.dto.page.CollItemSaveDto;
import com.qiho.center.api.enums.ItemStatusEnum;
import com.qiho.center.api.remoteservice.page.RemoteCollItemBackendService;
import com.qiho.center.api.util.JSONUtil;
import com.qiho.manager.biz.service.ItemService;
import com.qiho.manager.biz.service.page.CollItemService;
import com.qiho.manager.biz.vo.ItemVO;
import com.qiho.manager.biz.vo.page.CollItemVO;
import com.qiho.manager.common.param.SaveCollItemStyleParam;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author jiafeng.geng
 * @date 2018-07-19.
 */
@Service("collItemService")
public class CollItemServiceImpl implements CollItemService {

    private static final Logger LOGGER = LoggerFactory.getLogger(CollItemServiceImpl.class);

    private static final String COLL_ITEM_DEFAULT_SKIN_STYLE = "{\"styleType\" :1,\"styleId\":1}";

    private static final Integer COLL_ITEM_DEFAULT_STATUS = 0;

    @Resource
    private RemoteCollItemBackendService remoteCollItemBackendService;

    @Resource
    private ItemService itemService;

    @Override
    public List<CollItemVO> queryItemListBySort(Long collId) {
        // 查询集合页下的商品列表
        List<CollItemDto> collItemDtoList = remoteCollItemBackendService.queryItemListBySort(collId);
        if (CollectionUtils.isEmpty(collItemDtoList)) {
            LOGGER.info("要查询的集合页商品列表为空, collId={}", collId);
            return Lists.newArrayList();
        }

        // 过滤下架商品id
        List<Long> collItemIds = filterItemByStatus(collItemDtoList);

        // 根据集合页id和商品id集合查询集合页商品列表
        collItemDtoList = remoteCollItemBackendService.queryItemListBySortWithItemIds(collId, collItemIds);
        if (CollectionUtils.isEmpty(collItemDtoList)) {
            LOGGER.info("根据集合页id和商品id集合查询到的结果为空, collId={}, collItemIds={}", collId, collItemIds);
            return Lists.newArrayList();
        }

        // 最多展示50条
        int maxLen = 50;
        if (collItemDtoList.size() > maxLen) {
            collItemDtoList = collItemDtoList.subList(0, maxLen);
        }

        return transformCollItemVO(collItemDtoList);
    }

    private List<Long> filterItemByStatus(List<CollItemDto> collItemVOList) {
        // 提取商品id集合
        List<Long> itemIdList = collItemVOList.stream().map(CollItemDto::getId).collect(Collectors.toList());

        // 查询商品信息
        List<ItemVO> itemVOList = itemService.queryItemsByItemIds(itemIdList);

        List<Long> collItemIds = Lists.newArrayList();
        // 过滤已下架状态的商品id, 仅将上架状态的商品id返回
        for (ItemVO itemVO : itemVOList) {
            if (StringUtils.equals(itemVO.getItemStatus(), ItemStatusEnum.STATUS_ON.getCode())) {
                collItemIds.add(itemVO.getId());
            }
        }

        return collItemIds;
    }

    private List<CollItemVO> transformCollItemVO(List<CollItemDto> collItemDtoList) {
        List<CollItemVO> collItemVOList = Lists.newArrayList();
        // 转换商品配置信息为JSON对象
        collItemDtoList.forEach(collItemDto -> {
                    CollItemVO collItemVO = BeanUtils.copy(collItemDto, CollItemVO.class);
                    collItemVO.setStyleConfig(JSONUtil.strToJSON(collItemDto.getStyleConfig()));
                    collItemVOList.add(collItemVO);
                }
        );
        return collItemVOList;
    }

    @Override
    public Integer addCollItemList(Long collId, List<Long> itemIdList) {
        // 最多添加50条
        int size = remoteCollItemBackendService.queryItemListBySort(collId).size();
        int maxLen = 50;
        if (size > maxLen) {
            return 0;
        } else if (size + itemIdList.size() > maxLen) {
            itemIdList = itemIdList.subList(0, maxLen - size);
        }

        // 获取保存商品前的最大排序值
        Integer maxSort = remoteCollItemBackendService.findMaxSort(collId);
        if (null == maxSort) {
            maxSort = -1;
        }

        // 类型转换
        List<CollItemSaveDto> collItemSaveDtoList = Lists.newArrayList();
        for (Long itemId : itemIdList) {
            CollItemSaveDto collitemsaveDto = new CollItemSaveDto();
            collitemsaveDto.setId(collId);
            collitemsaveDto.setItemId(itemId);
            collitemsaveDto.setCollItemStatus(COLL_ITEM_DEFAULT_STATUS);
            collitemsaveDto.setStyleConfig(COLL_ITEM_DEFAULT_SKIN_STYLE);
            collitemsaveDto.setSort(++maxSort);
            collItemSaveDtoList.add(collitemsaveDto);
        }
        return remoteCollItemBackendService.addCollItemList(collItemSaveDtoList);
    }

    @Override
    public Integer deleted4Phy(Long collId, Long itemId) {
        return remoteCollItemBackendService.deleted4Phy(collId, itemId);
    }

    @Override
    public Integer sort(Long collId, List<Long> itemIds) {
        return remoteCollItemBackendService.sort(collId, itemIds);
    }

    @Override
    public Integer updateStatusBatch(Long collId, List<Long> itemIdList, Integer collItemStatus) {
        return remoteCollItemBackendService.updateStatusBatch(collId, itemIdList, collItemStatus);
    }

    @Override
    public Boolean checkStock(List<Long> itemIdList) {
        // 查询商品信息
        List<ItemVO> itemVOList = itemService.queryItemsByItemIds(itemIdList);
        if (CollectionUtils.isEmpty(itemVOList)) {
            LOGGER.info("要查询库存的集合页商品列表为空, itemIdList={}", itemIdList);
            return true;
        }

        // 遍历判断库存
        for (ItemVO itemVO : itemVOList) {
            if (itemVO.getStock() == 0) {
                LOGGER.info("当前集合页商品集合中包含库存为0的商品, itemIdList={}", itemIdList);
                return true;
            }
        }

        return false;
    }

    @Override
    public Integer saveStyle(SaveCollItemStyleParam param) {
        JSONObject styleConfigJson = param.getStyleConfig();
        // 判断样式信息是否为JSON格式
        if (!isJSON(styleConfigJson.toString())) {
            LOGGER.info("集合页商品样式格式有误, param={}", param);
            return null;
        }

        String styleConfig = styleConfigJson.toString();
        if (StringUtils.equals("0", styleConfigJson.get("styleType").toString())) {
            styleConfig = "";
        }

        // 查询集合页商品信息
        CollItemDto collItemDto = remoteCollItemBackendService.queryItem(param.getId(), param.getItemId());

        // 封装参数
        CollItemSaveDto collItemSaveDto = new CollItemSaveDto();
        collItemSaveDto.setItemId(param.getItemId());
        collItemSaveDto.setId(param.getId());
        collItemSaveDto.setStyleConfig(styleConfig);
        collItemSaveDto.setSort(collItemDto.getSort());
        return remoteCollItemBackendService.updateCollItemList(Lists.newArrayList(collItemSaveDto));
    }

    @Override
    public Integer batchDeleted(List<Long> itemIds) {
        if (CollectionUtils.isEmpty(itemIds)) {
            return -1;
        }
        return remoteCollItemBackendService.batchDeleted(itemIds);
    }

    /**
     * 判断字符串是否为json格式
     */
    private Boolean isJSON(String str) {
        try {
            JSONObject.fromObject(str);
            return true;
        } catch (Exception e) {
            return false;
        }
    }
}
