package com.qiho.manager.biz.service.template.impl;

import cn.com.duiba.message.service.api.dto.SmsSignDto;
import cn.com.duiba.message.service.api.dto.SmsTemplateDto;
import cn.com.duiba.message.service.api.remoteservice.RemoteSmsBackedService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.qiho.center.api.dto.TemplateDto;
import com.qiho.center.api.dto.merchant.MerchantDto;
import com.qiho.center.api.enums.SmsTypeEnum;
import com.qiho.center.api.params.template.TemplateParams;
import com.qiho.center.api.remoteservice.RemoteTemplateService;
import com.qiho.center.api.remoteservice.merchant.RemoteMerchantService;
import com.qiho.manager.biz.service.template.TemplateService;
import com.qiho.manager.biz.vo.Pagenation;
import com.qiho.manager.biz.vo.TemplateVO;
import com.qiho.manager.common.enums.*;
import com.qiho.manager.common.exception.QihoManagerException;
import com.qiho.manager.common.param.TemplatePageParam;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author Wangpf
 * @description
 * @date 2019/9/12 5:45 PM
 */
@Service
public class TemplateServiceImpl implements TemplateService {


    private static final Logger logger = LoggerFactory.getLogger(TemplateServiceImpl.class);

    @Autowired
    private RemoteMerchantService remoteMerchantService;

    @Autowired
    private RemoteTemplateService remoteTemplateService;

    @Autowired
    private StringRedisTemplate redisTemplate;

    @Autowired
    private RemoteSmsBackedService remoteSmsBackedService;

    /**
     * 连接符
     */
    private static final String JOINER = ",";

    /**
     * 敏感词连接符
     */
    private static final String JOINER_SENSITIVE = ";";

    /**
     * 绑定所有商家的
     */
    private static final String ALL_TEMPLATE = "-1";

    @Override
    public Pagenation<TemplateVO> findTemplateByParam(
        TemplatePageParam templatePageParam) {
        if (templatePageParam == null) {
            throw new IllegalArgumentException("Param can not be null");
        }
        String merchantIds = null;

        String merchantName = templatePageParam.getMerchantName();
        String templateName = templatePageParam.getTemplateName();
        String appId = StringUtils.remove(templatePageParam.getAppId(), " ");
        if (!StringUtils.isBlank(merchantName)) {
            //封装查询参数 将商家名字转换为ID
            List<MerchantDto> listByName = remoteMerchantService.findListByName(merchantName);
            if (!CollectionUtils.isEmpty(listByName)) {
                List<String> ids = listByName.stream().map(MerchantDto::getId).map(String::valueOf)
                    .collect(Collectors.toList());
                merchantIds = String.join(JOINER,ids);
            }
        }
        TemplateParams params = TemplateParams.builder().relatedMerchantIds(merchantIds)
                .templateName(templateName)
                .relatedAppIds(appId)
                .build();
        List<TemplateDto> list = remoteTemplateService.findTemplateByParam(params);
        list.forEach(e -> e.setSmsType(SmsTypeEnum.fromCode(e.getSmsType()) == null ?
            null :
            SmsTypeEnum.fromCode(e.getSmsType()).getMsg()));

        Pagenation<TemplateVO> result = new Pagenation<>();

        //封装对象模板绑定状态
        List<TemplateVO> templateVOS = BeanUtils.copyList(list,TemplateVO.class);
        if (CollectionUtils.isEmpty(templateVOS)) {
            return result.emptyPage();
        }

        // 模板绑定类型
        templateVOS.forEach(template -> {
            template.setAllBindSupport(SmsTemplateEnumExt.getBindSupport(template.getTemplateCode()));
            template.setAppBindSupport(SmsTemplateEnumExt.getAppBindSupport(template.getTemplateCode()));
            template.setBindType(queryTemplateBindType(template));
        });

        Integer pageNum = templatePageParam.getPageNum();
        //这里进行非空校验 不然会抛出npe
        pageNum = pageNum == null ? 1 : pageNum;
        Integer pageSize = templatePageParam.getPageSize();
        pageSize = pageSize == null ? 20 : pageSize;

        int total = templateVOS.size();
        //开始
        Integer start = pageNum <= 1 ? 0 : (pageNum - 1) * pageSize;

        //结束
        Integer end = start + pageSize;
        if (end >= total) {
            end = total;
        }

        // 会抛异常，特殊处理下
        if (start > end && start > total) {
            start = 0;
        }
        result.setList(templateVOS.subList(start,end));
        result.setTotal(total);
        return result;
    }

    @Override
    public boolean addTemplate(TemplateParams templateParams) {

        //校验短信模板敏感词
        checkSensitiveWord(templateParams);

        // 媒体定制短信参数校验
        checkAppBindParam(templateParams);
        // 商家绑定参数校验
        if (StringUtils.isBlank(templateParams.getRelatedAppIds())) {
            checkBindParam(templateParams);
        }

        //针对中台短信验证码的部分进行校验
        checkPlatFormSmsCode(templateParams);

        if (SmsTypeEnum.YUNPIAN == SmsTypeEnum.fromCode(templateParams.getSmsType())){
            templateParams.setNoteTemplateId(StringUtils.EMPTY);
        }

        DubboResult<Void> result = remoteTemplateService.addTemplate(templateParams);
        return result.isSuccess();
    }

    /**
     * 绑定中台短信验证  需要对签名Id 和 模版ID 进行校验
     *
     * @param templateParams
     */
    private void checkPlatFormSmsCode(TemplateParams templateParams) {
        if (SmsTypeEnum.CENTRAL_PLATFORM_CODE != SmsTypeEnum.fromCode(templateParams.getSmsType())){
            return;
        }

        //签名Id 和 短信Id 不能为空
        Long modelId = templateParams.getModelId();
        Long signId = templateParams.getSignId();
        if(modelId == null || signId == null){
            throw new QihoManagerException("中台短信模版Id/签名不能为空");
        }

        //校验签名Id 和短信Id
        SmsSignDto validSign = remoteSmsBackedService.getValidSign(signId);
        if(validSign == null ){
            throw new QihoManagerException("当前短信签名Id: " + signId + " 无效");
        }

        SmsTemplateDto validTemplate = remoteSmsBackedService.getValidTemplate(modelId);
        if(validTemplate == null){
            throw new QihoManagerException("当前短信模板Id: " + modelId + " 无效");
        }

        //中台短信验证码 只能绑定验证码/修改密码验证码/忘记密码验证码 三种类型
        String templateCode = remoteTemplateService.getTemplateCodeById(templateParams.getId());
        boolean isSmsCode = SmsTemplateEnumExt.SMS_CODE.getCode().equals(templateCode);
        boolean isForgetPwd = SmsTemplateEnumExt.FORGET_PWD.getCode().equals(templateCode);
        boolean isUpdatePwd = SmsTemplateEnumExt.UPDATE_PWD.getCode().equals(templateCode);
        if(!(isSmsCode || isForgetPwd || isUpdatePwd)){
            throw new QihoManagerException("当前短信模板代码: " + templateCode + " 不允许绑定该运营商: "+ SmsTypeEnum.CENTRAL_PLATFORM_CODE.getMsg());
        }
    }

    @Override
    public boolean updateTemplate(TemplateParams templateParams) {

        checkSensitiveWord(templateParams);

        // 媒体定制短信参数校验
        checkAppBindParam(templateParams);
        // 商家绑定参数校验
        if (StringUtils.isBlank(templateParams.getRelatedAppIds())) {
            checkBindParam(templateParams);
        }

        //针对中台短信验证码的部分进行校验
        checkPlatFormSmsCode(templateParams);

        return remoteTemplateService.updateTemplate(BeanUtils.copy(templateParams,TemplateParams.class)).isSuccess();
    }

    /**
     * @param templateParams
     */
    private void checkSensitiveWord(TemplateParams templateParams) {

        String sensitiveWord = redisTemplate.opsForValue().get(RedisKeyEnum.SENSITIVIE_WORD_KEY.getFinalKey());
        if(StringUtils.isEmpty(sensitiveWord)){
            return;
        }
        //短信模版内容
        String templateContext = templateParams.getTemplateContext();

        String[] sensitiveWords = sensitiveWord.split(JOINER_SENSITIVE);

        List<String> hitMessage = new ArrayList<>();

        for (String word : sensitiveWords) {
            if(templateContext.contains(word)){
                hitMessage.add(word);
            }
        }

        if(hitMessage.isEmpty()){
            return;
        }
        String errorMsg = "当前模版中 \"" + StringUtils.join(hitMessage,"、") + "\" 为禁用词汇，请修改后保存";

        throw new QihoManagerException(errorMsg);
    }

    private void checkBindParam(TemplateParams templateParams){

        if(templateParams == null){
            throw new IllegalArgumentException("Param can not be null");
        }

        //校验当前商家是否绑定过相同类型的短信模板
        String relatedMerchantIds = templateParams.getRelatedMerchantIds();
        String templateCode = templateParams.getTemplateCode();

        Long id = templateParams.getId();
        if(id != null){
            templateCode = remoteTemplateService.getTemplateCodeById(templateParams.getId());
        }

        //如果当前模板不支持部分商家绑定
        if(!SmsTemplateEnumExt.bindSupport(templateCode) && !Objects.equals(relatedMerchantIds,ALL_TEMPLATE)){
            throw new QihoManagerException(ErrorCode.NOT_ALLOWED.getMsg());
        }
        TemplateParams params = TemplateParams.builder()
            .relatedMerchantIds(relatedMerchantIds)
            .templateCode(templateCode).build();

        List<TemplateDto> templateDtoList = remoteTemplateService.findTemplateByParam(params);

        //这里将ID是自己的Id过滤掉
        List<TemplateDto> dtoList = templateDtoList.stream()
            .filter(templateDto ->!Objects.equals(id,templateDto.getId()))
            .collect(Collectors.toList());

        if(!CollectionUtils.isEmpty(dtoList)){
            //这里进行异常码的区分，前端进行不同文案的展示
            if(Objects.equals(relatedMerchantIds,ALL_TEMPLATE)){
                //单个模板类型绑定所有商家重复
                throw new QihoManagerException(ErrorCode.REPEAT.getMsg());
            }else {
                //单个商家重复绑定单个模板类型
                throw new QihoManagerException(ErrorCode.ONLY_ONE.getMsg());
            }
        }
    }

    /**
     * 媒体定制短信校验
     */
    private void checkAppBindParam(TemplateParams templateParams){
        if (templateParams == null) {
            throw new IllegalArgumentException("Param can not be null");
        }

        // 非媒体定制短信，无需校验
        if (StringUtils.isBlank(templateParams.getRelatedAppIds())) {
            return;
        }

        // 参数处理,去除多余空格和结尾分号
        templateParams.setRelatedAppIds(StringUtils.remove(templateParams.getRelatedAppIds(), " "));
        templateParams.setRelatedAppIds(StringUtils.removeEnd(templateParams.getRelatedAppIds(), ";"));

        //校验当前媒体是否绑定过相同类型的短信模板
        String relatedAppIds = templateParams.getRelatedAppIds();
        String templateCode = templateParams.getTemplateCode();

        Long id = templateParams.getId();
        if(id != null){
            templateCode = remoteTemplateService.getTemplateCodeById(templateParams.getId());
        }

        //如果当前模板不支持部分媒体绑定
        if (!SmsTemplateEnumExt.appBindSupport(templateCode)) {
            throw new QihoManagerException(ErrorCode.APP_NOT_ALLOWED.getMsg());
        }

        TemplateParams params = TemplateParams.builder()
                .relatedAppIds(relatedAppIds)
                .templateCode(templateCode).build();
        List<TemplateDto> templateDtoList = remoteTemplateService.findTemplateByParam(params);

        // 过滤自己的ID
        List<TemplateDto> dtoList = templateDtoList.stream()
                .filter(templateDto -> !Objects.equals(id, templateDto.getId()))
                .collect(Collectors.toList());

        // 检查已存在模板的媒体并返回错误信息
        if (!CollectionUtils.isEmpty(dtoList)) {
            Set<String> existAppIds = findAppExist(dtoList, relatedAppIds);
            String errMsg = Joiner.on("、")
                    .appendTo(new StringBuilder("媒体"), existAppIds)
                    .append("已存在订单通知模版")
                    .toString();
            throw new QihoManagerException(errMsg);
        }

        // 短信模板绑定的商家ID改为-2，即表示该模板绑定了部分媒体，在后续查询时不会被商家ID和全部商家(-1)作为条件查询出来
        templateParams.setRelatedMerchantIds(RelatedMerchantIdsEnum.APP.getType());
    }

    /**
     * 判断短信模板的绑定类型
     */
    private Integer queryTemplateBindType(TemplateVO template) {
        if (StringUtils.isNotBlank(template.getRelatedAppIds())) {
            return TemplateBindType.APP.getType();
        } else if (!CollectionUtils.isEmpty(template.getMerchantList())) {
            return TemplateBindType.MERCHANT.getType();
        } else {
            return TemplateBindType.TOTAL.getType();
        }
    }

    /**
     * 检查已经配置模板的媒体ID
     */
    private Set<String> findAppExist(List<TemplateDto> dtoList, String relatedAppIds) {
        // 获取已经配置模板的媒体ID集合
        Set<String> existAppIds = dtoList.stream()
                .filter(s -> StringUtils.isNotBlank(s.getRelatedAppIds()))
                .flatMap(s -> Splitter.on(";").omitEmptyStrings().trimResults().splitToList(s.getRelatedAppIds()).stream())
                .collect(Collectors.toSet());
        // 计算交集
        existAppIds.retainAll(Splitter.on(";").omitEmptyStrings().trimResults().splitToList(relatedAppIds));
        return existAppIds;
    }
}
