package com.qiho.manager.biz.process.component;

import cn.com.duiba.boot.profiler.DBTimeProfiler;
import com.google.common.io.CharSink;
import com.google.common.io.FileWriteMode;
import com.google.common.io.Files;
import com.qiho.center.api.dto.component.ComponentDto;
import com.qiho.center.api.remoteservice.component.RemoteComponentService;
import com.qiho.manager.biz.params.component.ComponentPagePreParam;
import com.qiho.manager.biz.runnable.DownloadComponentCodeRunnable;
import com.qiho.manager.biz.vo.component.ComponentCodeVO;
import com.qiho.manager.biz.vo.component.ComponentPagePreviewVO;
import com.qiho.manager.common.exception.QihoManagerException;
import com.qiho.manager.common.util.FileIDGenerator;
import com.qiho.manager.common.util.UploadTool;
import org.apache.commons.lang.StringUtils;
import org.apache.velocity.app.VelocityEngine;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;

import javax.annotation.Resource;
import java.io.File;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * 组件化页面的处理器
 *
 * @author chensong
 * @create 2018-09-20
 */
public abstract class ComponentPageProcessor {


    private static final Logger LOGGER = LoggerFactory.getLogger(ComponentPageProcessor.class);

    /** HTTP路径前缀 */
    protected static final String HTTP_URL_PRE = "http:";

    /** 上传文件路径前缀 */
    protected static final String PATH_PRE = "component/page/";

    /** DateTime时间格式 */
    protected static final String DATETIME_FORMAT = "yyyyMMdd";

    /** url路径分隔符 */
    protected static final String URL_SEPARATOR = "/";

    /** 作为Map的key使用 Map key Start  */

    protected static final String HTML_URL_KEY = "htmlUrl";

    protected static final String HTML_MD5_KEY = "htmlMd5";

    protected static final String CSS_URL_KEY = "cssUrl";

    protected static final String JS_URL_KEY = "jsUrl";

    protected static final String HTML_CODE_KEY = "htmlCode";

    /** Map Key End  */


    @Value("${oss.objectPath}")
    protected String ossYunUrl;

    @Autowired
    protected VelocityEngine velocityEngine;

    @Resource
    protected ExecutorService executorService;

    @Autowired
    protected RemoteComponentService remoteComponentService;

    @Autowired
    protected ApplicationContext applicationContext;

    /**
     * 组件化页面预览
     * 校验各个组件是否有效  组件必填项是否存在
     * 将各个组件的CSS代码打包成一个文件 JS代码打包成一个文件
     * 替换pageShell.vm 的占位符 生成将新的HTML代码 获取MD5摘要 将HTML代码上传到Oss
     * 返回前端预览HTML文件的url  预览地址  MD5摘要
     * @param param
     * @return
     */
    public abstract ComponentPagePreviewVO preview(ComponentPagePreParam param);

    /**
     * 从Oss服务器下载代码
     * 返回组件id以及组件代码对象的Map
     *
     * @param componentDtoList
     * @return Map -> (key：组件id  value：组件代码对象)
     */
    @DBTimeProfiler
    protected Map<Long, ComponentCodeVO> downloadCode(List<ComponentDto> componentDtoList){

        // 用于存储代码  key->组件id  value->组件代码对象
        Map<Long, ComponentCodeVO> resultMap = new ConcurrentHashMap<>(16);
        CountDownLatch latch = new CountDownLatch(componentDtoList.size());

        // 多线程下载代码 然后解析
        for (ComponentDto componentDto: componentDtoList) {
            DownloadComponentCodeRunnable runnable =
                    applicationContext.getBean(DownloadComponentCodeRunnable.class);
            runnable.setComponentId(componentDto.getId());
            runnable.setFileUrl(componentDto.getCodeUrl());
            runnable.setLatch(latch);
            runnable.setResultMap(resultMap);
            executorService.submit(runnable);
        }

        try {
            // 最大等待20s
            latch.await(20, TimeUnit.SECONDS);
        } catch (Exception e) {
            LOGGER.error("代码解析错误，data = {} ", componentDtoList, e);
            throw new QihoManagerException("组件代码解析错误," + e.getMessage());
        }

        if (resultMap.size() != componentDtoList.size()) {
            throw new QihoManagerException("组件代码解析错误");
        }

        return resultMap;
    }



    /**
     * 将css js代码分别组成一个文件 上传到CDN
     * 返回css文件地址 Js文件地址
     * 将HTML代码拼接返回
     *
     * @param codeMap
     * @return css代码文件url js代码文件url html代码
     */
    protected Map<String, String> buildCodeFile(Map<Long, ComponentCodeVO> codeMap, List<Long> idList){
        StringBuilder htmlBuilder = new StringBuilder();
        StringBuilder cssBuilder = new StringBuilder();
        StringBuilder jsBuilder = new StringBuilder();

        for (Long id : idList) {
            ComponentCodeVO componentCode = codeMap.get(id);
            if (componentCode == null) {
                LOGGER.error("代码拼接失败，组件id = [{}] 为null", id);
                throw new QihoManagerException("代码组合失败, 组件id=" + id + "代码不存在");
            }
            htmlBuilder.append(skipBlank(componentCode.getHtmlCode())).append("\r\n");
            cssBuilder.append(skipBlank(componentCode.getCssCode()));
            jsBuilder.append(skipBlank(componentCode.getJsCode()));
        }

        // 返回结果对象
        Map<String, String> resultMap = new HashMap<>(16);

        // 打包上传Css文件
        uploadCssFile(resultMap, cssBuilder.toString(), idList);

        // 打包上传Js文件
        uploadJsFile(resultMap, jsBuilder.toString(), idList);

        // put Html代码
        resultMap.put(HTML_CODE_KEY, htmlBuilder.toString());

        return resultMap;
    }


    /**
     * 略过空串和null
     * @param str
     * @return
     */
    private String skipBlank(String str){
        return StringUtils.isBlank(str) ? StringUtils.EMPTY : str;
    }


    /**
     * 打包上传Css文件
     * @param resultMap
     * @param cssCode
     */
    protected void uploadCssFile(Map<String, String> resultMap, String cssCode, List<Long> idList){
        if (StringUtils.isBlank(cssCode)) {
            LOGGER.error("css代码为空，组件id={}", idList);
            throw new QihoManagerException("Css代码为空");
        }

        String cssFileName = FileIDGenerator.getRandomString(10);
        try {
            File cssFile = File.createTempFile(cssFileName,".css");
            CharSink charSink = Files.asCharSink(cssFile, Charset.defaultCharset(), FileWriteMode.APPEND);
            charSink.write(cssCode);

            String cssFilePath = PATH_PRE + new DateTime().toString(DATETIME_FORMAT) + URL_SEPARATOR +cssFile.getName();
            boolean cssResult = UploadTool.uploadOssCss(cssFile, cssFilePath);
            if (!cssResult) {
                throw new QihoManagerException("上传Css文件失败");
            }

            // put Css文件地址
            resultMap.put(CSS_URL_KEY, ossYunUrl + cssFilePath);

        } catch (Exception e) {
            LOGGER.error("上传CSS文件异常", e);
            throw new QihoManagerException(e.getMessage());
        }
    }

    /**
     * 打包上传js文件
     * @param resultMap
     * @param jsCode
     * @param idList
     */
    protected void uploadJsFile(Map<String, String> resultMap, String jsCode, List<Long> idList){
        if (StringUtils.isBlank(jsCode)) {
            LOGGER.error("Js代码为空，组件id={}", idList);
            throw new QihoManagerException("Js代码为空");
        }
        String jsFileName = FileIDGenerator.getRandomString(10);
        try {
            File cssFile = File.createTempFile(jsFileName,".js");
            CharSink charSink = Files.asCharSink(cssFile, Charset.defaultCharset(), FileWriteMode.APPEND);
            charSink.write(jsCode);

            String cssFilePath = PATH_PRE + new DateTime().toString(DATETIME_FORMAT) + URL_SEPARATOR +cssFile.getName();
            boolean jsResult = UploadTool.uploadOssCss(cssFile, cssFilePath);
            if (!jsResult) {
                throw new QihoManagerException("上传Js文件失败");
            }

            // put Js文件地址
            resultMap.put(JS_URL_KEY, ossYunUrl + cssFilePath);

        } catch (Exception e) {
            LOGGER.error("上传JS文件异常", e);
            throw new QihoManagerException(e.getMessage());
        }
    }











}
