package com.qiho.manager.biz.runnable;

import cn.afterturn.easypoi.excel.ExcelExportUtil;
import cn.afterturn.easypoi.excel.entity.ExportParams;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.collect.Lists;
import com.qiho.center.api.dto.OrderSnapshotDto;
import com.qiho.center.api.dto.es.OrderSnapshotESDto;
import com.qiho.center.api.dto.reply.UserReplyDto;
import com.qiho.center.api.params.OrderPageParams;
import com.qiho.center.api.remoteservice.order.RemoteOrderService;
import com.qiho.manager.biz.params.OrdersExportParam;
import com.qiho.manager.common.constant.CacheConstantseEnum;
import com.qiho.manager.common.util.UploadTool;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.ss.usermodel.Workbook;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.data.redis.core.HashOperations;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.Map;

/**
 * Created by liuyao on 2017/6/7.
 */
@Component
@Scope("prototype")
public class OrdersExportRunnable extends BaseExportRunnable implements Runnable {


    private static final Logger LOGGER = LoggerFactory.getLogger(OrdersExportRunnable.class);

    private static final int ORDER_EXPORT_MAX = 1000;

    @Autowired
    private RemoteOrderService remoteOrderService;

    @Resource(name = "redisTemplate")
    private AdvancedCacheClient advancedCacheClient;

    private OrderPageParams params;

    private File file;

    private String filePath = "orders/";

    private Long taskId = System.currentTimeMillis();

    /**
     * 初始化
     *
     * @return 任务号
     */
    public Map<String, Object> init() {
        file = super.getFile2(filePath);
        return super.init(file, taskId);
    }

    @Override
    public void run() {

        Integer exportCount = remoteOrderService.exportOrderESCount(params);
        int batch = getBatchs(exportCount);
        String redisKey = CacheConstantseEnum.EXPORT_ORDER_CACEL + ":" + taskId;

        CacheConstantseEnum constantseEnum = CacheConstantseEnum.EXPORT_ORDER;
        String key = constantseEnum.getCacheKey(taskId);
        HashOperations<String, String, Object> operations = redisTemplate.opsForHash();
        Workbook workbook = null;
        try (FileOutputStream outputStream = new FileOutputStream(file)) {
            List<OrdersExportParam> exportList = Lists.newArrayList();
            for(int i=0;i<batch;i++) {
                String redisTaskId = advancedCacheClient.get(redisKey);
                if (StringUtils.isNotBlank(redisTaskId)) {
                    redisTemplate.delete(key);
                    advancedCacheClient.remove(redisKey);
                    return;
                }

                params.setOffset(i*ORDER_EXPORT_MAX);
                OrderSnapshotESDto result = remoteOrderService.exportAllOrderPage(params);
                params.setScrollId(result.getScrollId());
                List<OrdersExportParam> list = transferExcelEntity(result.getList());
                exportList.addAll(list);
                operations.increment(key,"successCount",list.size());
            }
            workbook = ExcelExportUtil.exportBigExcel(new ExportParams(), OrdersExportParam.class, exportList);
            workbook.write(outputStream);
            operations.put(key, "message", "文件上传中");
            String url = UploadTool.uploadOssNotCDN(file,
                    filePath + file.getName(), "application/x-msdownload;charset=GBK");

            operations.put(key, "url", url);
            operations.put(key, "message", "导出成功");
        } catch (Exception e) {
            LOGGER.error("orders Export error msg={}", e);
            operations.put(key, "message", e.getMessage());
        } finally {
            operations.put(key, "success", "true");
            ExcelExportUtil.closeExportBigExcel();
            try {
                workbook.close();
            } catch (IOException e) {
                LOGGER.error("资源close异常", e);
            }
            if (file != null && file.exists() && !file.delete()) {
                LOGGER.warn("file delete error or file not exists");
            }
        }
    }

    /**
     * 转换成Excel导出的格式
     *
     * @param result
     * @return
     */
    private List<OrdersExportParam> transferExcelEntity(List<OrderSnapshotDto> result) {

        List<OrdersExportParam> list = Lists.newArrayList();
        for (OrderSnapshotDto dto : result) {
            OrdersExportParam ordersExport = BeanUtils.copy(dto, OrdersExportParam.class);

            try {
                ordersExport.setOrderAmt(dto.getOrderAmt() / 100.0);
            } catch (Exception e) {
                LOGGER.info("批量导出订单-设置订单金额错误, dto={}", dto);
            }

            try {
                ordersExport.setSellingPrice(dto.getSellingPrice() / 100.0);
            } catch (Exception e) {
                LOGGER.info("批量导出订单-设置商品价格错误, dto={}", dto);
            }

            if (dto.getDeliveryEnum() == null) {
                ordersExport.setDelivery("");
            } else {
                ordersExport.setDelivery(dto.getDeliveryEnum().getDesc());
            }
            ordersExport.setOrderStatus(dto.getOrderStatusEnum().getDesc());
            String reply = convertReplyToString(dto.getUserReplyList());
            ordersExport.setUserReply(reply);
            list.add(ordersExport);
        }

        return list;
    }

    /**
     * 将用户回复转成String类型
     *
     * @param userReplyList
     * @return
     */
    private String convertReplyToString(List<UserReplyDto> userReplyList) {
        if (CollectionUtils.isEmpty(userReplyList)) {
            return null;
        }
        StringBuilder userReplyBuilder = new StringBuilder();
        userReplyList.stream().forEach(e -> {
            userReplyBuilder.append(new DateTime(e.getGmtCreate()).toString("yyyy.MM.dd HH:mm:ss"))
                    .append(" ").append(e.getContent()).append("|");
        });
        String userReplyStr = userReplyBuilder.toString();
        // 去除最后一位 "|"
        return userReplyStr.substring(0, userReplyStr.length() - 1);
    }


    private int getBatchs(int total) {
        return total % ORDER_EXPORT_MAX == 0 ? total / ORDER_EXPORT_MAX : (total / ORDER_EXPORT_MAX + 1);
    }

    public OrderPageParams getParams() {
        return params;
    }

    public void setParams(OrderPageParams params) {
        params.setMax(ORDER_EXPORT_MAX);
        params.setOffset(0);
        this.params = params;
    }

}
