package com.qiho.manager.biz.service.advert.impl;

import cn.com.duiba.sso.api.tool.RequestTool;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.qiho.center.api.dto.ResultDto;
import com.qiho.center.api.dto.advert.CollectionAdvertDto;
import com.qiho.center.api.remoteservice.advert.RemoteCollectionAdvertService;
import com.qiho.center.api.remoteservice.page.RemoteCollPageBackendService;
import com.qiho.manager.biz.params.advert.CollectionAdvertSaveParam;
import com.qiho.manager.biz.service.advert.CollectionAdvertService;
import com.qiho.manager.biz.vo.advert.CollectionAdvertVO;
import com.qiho.manager.common.exception.QihoManagerException;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import static com.qiho.manager.common.util.AssertUtil.collectionOnRange;
import static com.qiho.manager.common.util.AssertUtil.numericIsPositive;

/**
 * @author chensong
 * @create 2018-09-13
 */
@Service
public class CollectionAdvertServiceImpl implements CollectionAdvertService {

    @Autowired
    private RemoteCollectionAdvertService remoteCollectionAdvertService;

    @Autowired
    private RemoteCollPageBackendService remoteCollPageBackendService;

    @Override
    public Boolean saveCollectionAdvert(CollectionAdvertSaveParam param) {

        // 参数校验
        numericIsPositive(param.getCollectionId(), "集合页id不合法");
        collectionOnRange(param.getAdvertIds(), 1, 5, "广告位id限定在1-5个");
        checkRepeat(param);

        // 新增数据
        if (param.getId() == null) {
            return createCollectionAdvert(param);
        }

        // 修改数据
        return updateCollectionAdvert(param);
    }

    /**
     * 广告位id重复性校验
     * @param param
     */
    private void checkRepeat(CollectionAdvertSaveParam param) {
        List<Long> advertIdList = param.getAdvertIds();
        Set<Long> advertIdSet = new HashSet<>(advertIdList);
        if (advertIdList.size() == advertIdSet.size()) {
            return;
        }

        advertIdList = advertIdList.stream()
                .collect(Collectors.toMap(e -> e, e -> 1, (a, b) -> a + b))
                .entrySet().stream()
                .filter(entry -> entry.getValue() > 1)
                .map(entry -> entry.getKey())
                .collect(Collectors.toList());

        throw new QihoManagerException("广告位id=" + advertIdList.toString() + "存在重复");
    }


    @Override
    public List<CollectionAdvertVO> listAllCollectionAdvert() {

        List<CollectionAdvertDto> list = remoteCollectionAdvertService.listAllCollectionAdvert();
        if (CollectionUtils.isEmpty(list)) {
            return Lists.newArrayList();
        }

        List<CollectionAdvertVO> voList = new ArrayList<>();
        Joiner joiner = Joiner.on(" ");
        for (CollectionAdvertDto e : list) {
            CollectionAdvertVO vo = BeanUtils.copy(e, CollectionAdvertVO.class);
            vo.setAdvertIdString(joiner.join(e.getAdvertIds()));
            voList.add(vo);
        }

        return voList;
    }

    @Override
    public Boolean delete(Long id) {

        numericIsPositive(id, "id不合法");

        CollectionAdvertDto collectionAdvert = new CollectionAdvertDto();
        collectionAdvert.setId(id);
        collectionAdvert.setOpeartor(RequestTool.getAdmin().getName());

        ResultDto<Boolean> result = remoteCollectionAdvertService.deletedCollectionAdvert(collectionAdvert);
        if (!result.isSuccess()) {
            throw new QihoManagerException(result.getMsg());
        }

        return result.getResult();
    }

    @Override
    public CollectionAdvertVO findById(Long id) {
        if (id == null || id < 1L) {
            return null;
        }
        return BeanUtils.copy(remoteCollectionAdvertService.findById(id), CollectionAdvertVO.class);
    }

    /**
     * 新增集合页广告位
     * @param param
     * @return
     */
    private Boolean createCollectionAdvert(CollectionAdvertSaveParam param) {

        // 校验集合页id是否存在
        if (remoteCollPageBackendService.findById(param.getCollectionId()) == null){
            throw new QihoManagerException("请输入正确的集合页ID");
        }

        // 校验该集合页id是否存在广告位记录
        if (remoteCollectionAdvertService.findByCollectionId(param.getCollectionId()) != null) {
            throw new QihoManagerException("该集合页ID已存在");
        }

        CollectionAdvertDto collectionAdvert = new CollectionAdvertDto();
        collectionAdvert.setCollectionId(param.getCollectionId());
        collectionAdvert.setAdvertIds(param.getAdvertIds());
        collectionAdvert.setOpeartor(RequestTool.getAdmin().getName());

        ResultDto<Boolean> result = remoteCollectionAdvertService.insertCollectionAdvert(collectionAdvert);
        if (!result.isSuccess()) {
            throw new QihoManagerException(StringUtils.defaultString(result.getMsg()));
        }

        return result.getResult();
    }

    /**
     * 修改集合页广告位
     * @param param
     * @return
     */
    private Boolean updateCollectionAdvert(CollectionAdvertSaveParam param) {

        numericIsPositive(param.getId(), "集合页id不合法");

        CollectionAdvertDto collectionAdvert = new CollectionAdvertDto();
        collectionAdvert.setId(param.getId());
        collectionAdvert.setCollectionId(param.getCollectionId());
        collectionAdvert.setAdvertIds(param.getAdvertIds());
        collectionAdvert.setOpeartor(RequestTool.getAdmin().getName());

        ResultDto<Boolean> result = remoteCollectionAdvertService.updateCollectionAdvert(collectionAdvert);
        if (!result.isSuccess()) {
            throw new QihoManagerException(StringUtils.defaultString(result.getMsg()));
        }

        return result.getResult();
    }


}
