/**
 * Project Name:qiho-manager-biz File Name:ItemServiceImpl.java Package Name:com.qiho.manager.biz.service.impl
 * Date:2017年5月23日上午10:28:26 Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 */

package com.qiho.manager.biz.service.impl;

import cn.com.duiba.sso.api.domain.dto.AdminDto;
import cn.com.duiba.sso.api.remoteservice.RemoteAdminService;
import cn.com.duiba.sso.api.tool.RequestTool;
import cn.com.duiba.sso.api.web.power.AdminPowerCacheService;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.DateUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.google.common.collect.Lists;
import com.qiho.center.api.constant.ItemConstants;
import com.qiho.center.api.dto.*;
import com.qiho.center.api.enums.DeliveryEnum;
import com.qiho.center.api.enums.ItemStatusEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.remoteservice.RemoteItemService;
import com.qiho.center.api.remoteservice.RemoteRecommendItemService;
import com.qiho.center.api.remoteservice.merchant.RemoteItemMerchantService;
import com.qiho.center.api.remoteservice.tag.RemoteTagService;
import com.qiho.manager.biz.params.ItemAdvertParam;
import com.qiho.manager.biz.service.ItemService;
import com.qiho.manager.biz.vo.*;
import com.qiho.manager.common.constant.CacheConstantseEnum;
import com.qiho.manager.common.constant.DomainConstantUtil;
import com.qiho.manager.common.exception.QihoManagerException;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

/**
 * ClassName:ItemServiceImpl <br/>
 * Date: 2017年5月23日 上午10:28:26 <br/>
 *
 * @author zhanglihui
 * @see
 * @since JDK 1.6
 */
@Service
public class ItemServiceImpl implements ItemService {

    /**
     * 默认商品限购数量
     */
    private static final Integer DEFAULT_LIMIT_NUMBER = 3;

    private static final Logger LOGGER = LoggerFactory.getLogger(ItemServiceImpl.class);

    @Autowired
    private RemoteItemService remoteItemService;

    @Autowired
    private RemoteAdminService remoteAdminService;

    @Autowired
    private RemoteRecommendItemService remoteRecommendItemService;

    @Autowired
    private RemoteItemMerchantService remoteItemMerchantService;

    @Autowired
    private RemoteTagService remoteTagService;

    @Resource(name = "redisTemplate")
    private AdvancedCacheClient advancedCacheClient;

    @Autowired
    private AdminPowerCacheService adminPowerCacheService;

    @Override
    public Pagenation<ItemVO> queryList(ItemDto queryItem, Integer pageNum, Integer pageSize) {

        boolean flag = adminPowerCacheService.hasPower(RequestTool.getAdminId(), "seeAllMerchant");
        Long ownerId = flag ? null : RequestTool.getAdminId();

        DubboResult<PagenationDto<ItemDto>> result = remoteItemService.queryItemPage(queryItem, ownerId, pageNum, pageSize);
        PagenationDto<ItemDto> pageDto = result.getResult();
        List<ItemDto> itemDtoList = pageDto.getList();
        Pagenation<ItemVO> page = new Pagenation<>();
        page.setTotal(pageDto.getTotal());
        List<ItemVO> itemVOList = transformItemVO(itemDtoList);
        page.setList(itemVOList);

        return page;
    }

    @Override
    public ItemDetailVO queryItemDetail(Long itemId) {

        DubboResult<ItemDetailDto> result = remoteItemService.queryItemDetail(itemId);

        ItemDetailDto itemDetailDto = result.getResult();
        ItemDetailVO itemDetailVO = BeanUtils.copy(result.getResult(), ItemDetailVO.class);
        itemDetailVO.setTopCategoryName(itemDetailDto.getExtParamValue(ItemVO.TOP_CATEGORY_NAME));
        itemDetailVO.setCategoryIdList(
            JSON.parseObject(itemDetailDto.getExtParamValue(ItemVO.CATEGORY_ID_LIST), new TypeReference<List<Long>>() {
            }));
        itemDetailVO.setCountDownCycle(itemDetailDto.getExtParamValue(ItemVO.COUNT_DOWN_CYCLE));
        itemDetailVO.setCountDownLimit(itemDetailDto.getExtParamValue(ItemVO.COUNT_DOWN_LIMIT));
        if (itemDetailDto.getMerchantDto() != null) {
            itemDetailVO.setMerchantId(itemDetailDto.getMerchantDto().getId());
            itemDetailVO.setMerchantDelivery(DeliveryEnum.getByCode(itemDetailDto.getDelivery()).getDesc());
            itemDetailVO.setMerchantName(itemDetailDto.getMerchantDto().getMerchantName());
        }

        itemDetailVO.setImage(itemDetailDto.getExtParamValue(ItemVO.IMAGESTR));
        if (itemDetailDto.getExtParamValue(ItemVO.LIMIT_NUMBER) == null) {
            itemDetailVO.setLimitNumber(DEFAULT_LIMIT_NUMBER);
        } else {
            itemDetailVO.setLimitNumber(Integer.valueOf(itemDetailDto.getExtParamValue(ItemVO.LIMIT_NUMBER)));
        }

        // 详情图片高度
        if (StringUtils.isBlank(itemDetailDto.getExtParamValue(ItemVO.IMG_HEIGHT))) {
            itemDetailVO.setImgHeight(Lists.newArrayList());
        } else {
            List<ItemImgVO> imgHeightList = JSON
                .parseArray(itemDetailDto.getExtParamValue(ItemVO.IMG_HEIGHT), ItemImgVO.class);
            itemDetailVO.setImgHeight(imgHeightList);
        }

        // 商品主图地址 商品介绍 营销标签
        itemDetailVO.setMainImgUrl(itemDetailDto.getExtParamValue(ItemVO.MAIN_IMG_URL));
        itemDetailVO.setItemIntroduce(itemDetailDto.getExtParamValue(ItemVO.ITEM_INTRODUCE));
        itemDetailVO.setSaleLableUrl(itemDetailDto.getExtParamValue(ItemVO.SALE_LABLE_URL));

        String supportCOD = itemDetailDto.getExtParamValue(ItemVO.SUPPORT_COD);
        //货到付款标记
        itemDetailVO.setSupportCOD(
            StringUtils.isBlank(supportCOD) || StringUtils.equals(StringUtils.trimToNull(supportCOD), "1") ?
                Boolean.TRUE :
                Boolean.FALSE);
        itemDetailVO.setSkuList(converToItemSkuVO(itemDetailDto.getSkuList()));
        itemDetailVO.setSkuProperties(BeanUtils.copyList(itemDetailDto.getSkuPropertyList(), SkuPropertyVO.class));
        itemDetailVO.setGmtModified(DateUtils.getSecondStr(itemDetailDto.getGmtModified()));
        try{
            AdminDto admin = remoteAdminService.findOneAdmin(itemDetailDto.getGmtModifier());
            itemDetailVO.setGmtModifyEmail(admin.getEmail());
            itemDetailVO.setGmtModifyName(admin.getName());
        } catch (Exception e){
            LOGGER.warn("获取管理员异常，itemId: {} ", itemId, e);
            itemDetailVO.setGmtModifyEmail(StringUtils.EMPTY);
            itemDetailVO.setGmtModifyName(StringUtils.EMPTY);
        }
        itemDetailVO
            .setMainRecomIds(itemDetailDto.getExtParamValue(ItemConstants.ItemExtConstans.MAIN_RECOMMEND_ITEM_ID));
        return itemDetailVO;
    }

    private List<ItemSkuVO> converToItemSkuVO(List<ItemSkuDto> list) {
        List<ItemSkuVO> voList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(list)) {
            return voList;
        }
        for (ItemSkuDto dto : list) {
            ItemSkuVO vo = BeanUtils.copy(dto, ItemSkuVO.class);
            vo.setStockId(dto.getStockId().toString());
            voList.add(vo);
        }
        return voList;
    }

    @Override
    public Long saveItem(ItemDetailDto itemDetailDto) {
        Long itemId = itemDetailDto.getId();
        itemDetailDto.setGmtModifier(RequestTool.getAdminId());
        if (itemId == null) {
            itemDetailDto.setGmtCreator(RequestTool.getAdminId());
            DubboResult<Long> result = remoteItemService.createItem(itemDetailDto);
            if (!result.isSuccess()) {
                throw new QihoException(result.getMsg());
            }
            itemId = result.getResult();
        } else {
            DubboResult<Void> result = remoteItemService.updateItem(itemDetailDto);
            if (!result.isSuccess()) {
                throw new QihoException(result.getMsg());
            }

            // 从缓存中清除商品详情的缓存
            String key = CacheConstantseEnum.ITEM_ID + itemId.toString();
            advancedCacheClient.remove(key);

        }
        return itemId;
    }

    @Override
    public void updateStatusBatch(List<Long> itemIdList, String status, Long gmtModifier) {
        remoteItemService.updateStatus(itemIdList, status, gmtModifier);
    }

    @Override
    public void deleteBatch(List<Long> itemIdList, Long gmtModifier) {
        remoteItemService.deleteBatch(itemIdList, gmtModifier);
        remoteItemMerchantService.deleteBatchByItemId(itemIdList, gmtModifier);
    }

    @Override
    public void recommendBatch(List<Long> itemIdList) {
        remoteRecommendItemService.recommendBatch(itemIdList);
    }

    @Override
    public List<ItemVO> transformItemVO(List<ItemDto> itemDtos) {

        return Lists.transform(itemDtos, itemDto -> {
            ItemVO itemVO = BeanUtils.copy(itemDto, ItemVO.class);
            itemVO.setTopCategoryName(itemDto.getExtParamValue("topCategoryName"));
            itemVO.setImage(itemDto.getExtParamValue("image"));
            // 设置商品详情URL
            itemVO.setUrl(DomainConstantUtil.getQihoWebUrl() + "/item/detail?id=" + itemDto.getId());
            return itemVO;
        });
    }

    @Override
    public List<ItemVO> querySimpleList(ItemDto itemDto) {
        if (StringUtils.isEmpty(itemDto.getItemStatus())) {
            itemDto.setItemStatus(ItemStatusEnum.STATUS_ON.getCode());
        }

        DubboResult<List<ItemSimpleDto>> result = remoteItemService.queryItemSimpleList(itemDto);
        List<ItemSimpleDto> dtoList = result.getResult();
        return BeanUtils.copyList(dtoList, ItemVO.class);
    }

    @Override
    public Boolean updateItemAdvertIdsAndthreshold(ItemAdvertParam param) {
        DubboResult<ItemAdvertDto> resultDto = remoteItemService.queryExtParamByItemId(param.getItemId());
        if (!resultDto.isSuccess() || null == resultDto.getResult())
            throw new QihoManagerException("itemId 查询不到对应的商品扩展信息");
        ItemAdvertDto itemAdvertDto = resultDto.getResult();
        String extParamStr = itemAdvertDto.getExtParam();
        JSONObject extJson = JSONObject.parseObject(extParamStr);
        if (null == extJson)
            extJson = new JSONObject();
        extJson.put(ItemConstants.ItemExtConstans.ADVERTIDS, param.getAdvertIds());
        extJson.put(ItemConstants.ItemExtConstans.THRESHOLD, param.getThreshold());
        itemAdvertDto.setExtParam(extJson.toJSONString());
        DubboResult<Boolean> resultBase = remoteItemService.updateExtParamByItemId(itemAdvertDto);
        return resultBase.isSuccess() && resultBase.getResult();
    }

    @Override
    public ItemAdvertVo queryItemAdvertByItemId(Long itemId) {
        ItemAdvertVo itemAdvertVo = new ItemAdvertVo();
        itemAdvertVo.setItemId(itemId);
        DubboResult<ItemAdvertDto> result = remoteItemService.queryExtParamByItemId(itemId);
        if (!result.isSuccess() || null == result.getResult())
            return itemAdvertVo;
        ItemAdvertDto dto = result.getResult();
        JSONObject extJson = JSONObject.parseObject(dto.getExtParam());
        if (null == extJson)
            return itemAdvertVo;
        itemAdvertVo.setAdvertIds(extJson.getString(ItemConstants.ItemExtConstans.ADVERTIDS));
        itemAdvertVo.setThreshold(extJson.getString(ItemConstants.ItemExtConstans.THRESHOLD));
        return itemAdvertVo;
    }

    @Override
    public ItemAdvertDto queryExtParamByItemId(Long itemId) {
        DubboResult<ItemAdvertDto> dubboResult = remoteItemService.queryExtParamByItemId(itemId);
        if (dubboResult.isSuccess() && null != dubboResult.getResult())
            return dubboResult.getResult();
        return new ItemAdvertDto();
    }

    @Override
    public Boolean updateExtParamByItemId(Long itemId, Map<String, String> extParam) {
        DubboResult<ItemAdvertDto> dubboResult = remoteItemService.queryExtParamByItemId(itemId);
        if (!dubboResult.isSuccess() || null == dubboResult.getResult())
            throw new QihoException("查询不到需要更新的商品的配置");
        String extParamStr = dubboResult.getResult().getExtParam();
        JSONObject extJson = JSONObject.parseObject(extParamStr);
        if (null == extJson)
            extJson = new JSONObject();
        for (Map.Entry<String, String> entry : extParam.entrySet()) {
            extJson.put(entry.getKey(), entry.getValue());
        }
        ItemAdvertDto updateParam = new ItemAdvertDto();
        updateParam.setItemId(itemId);
        updateParam.setExtParam(extJson.toJSONString());
        remoteItemService.updateExtParamByItemId(updateParam);
        remoteItemService.updateVirtailStock(itemId);
        return Boolean.TRUE;
    }

    @Override
    public List<ItemVO> queryItemsByItemIds(List<Long> itemIdList) {
        List<ItemDto> itemDtoList = remoteItemService.queryItemsByItemIds(itemIdList);
        return BeanUtils.copyList(itemDtoList, ItemVO.class);
    }
}
