package com.qiho.manager.biz.service.impl;

import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.DateUtils;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.logistics.LogisticsOrderDto;
import com.qiho.center.api.dto.logistics.LogisticsProcessDto;
import com.qiho.center.api.dto.logistics.QueryLogisticsOrderDto;
import com.qiho.center.api.dto.logistics.QueryTYOLogisticsMsgDto;
import com.qiho.center.api.enums.*;
import com.qiho.center.api.remoteservice.RemoteLogisticsOrderService;
import com.qiho.manager.biz.params.BatchAuditOrderParam;
import com.qiho.manager.biz.runnable.DistributeOrderExportRunnable;
import com.qiho.manager.biz.runnable.ordertaskhandler.LogisticsOrderSignHandler;
import com.qiho.manager.biz.service.LogisticsOrderService;
import com.qiho.manager.biz.service.OrderDealWithService;
import com.qiho.manager.biz.vo.OrderDealWithVO;
import com.qiho.manager.biz.vo.Pagenation;
import com.qiho.manager.biz.vo.logistics.LogisticsOrderVO;
import com.qiho.manager.biz.vo.logistics.YTOLogisticsReturnVO;
import com.qiho.manager.common.constant.CacheConstantseEnum;
import com.qiho.manager.common.constant.YTOInterfaceUtil;
import com.qiho.manager.common.exception.QihoManagerException;
import com.qiho.manager.common.util.AppLogUtil;
import com.qiho.manager.common.util.Environment;
import com.qiho.manager.common.util.JaxbUtil;
import com.qiho.manager.common.util.MessageDigestUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.StringEscapeUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutorService;

@Service
public class LogisticsOrderServiceImpl implements LogisticsOrderService,ApplicationContextAware {

    private static final Logger         LOG = LoggerFactory.getLogger(LogisticsOrderServiceImpl.class);

    @Autowired
    private RemoteLogisticsOrderService remoteLogisticsOrderService;

    @Autowired
    private OrderDealWithService        orderDealWithService;

    @Autowired
    private ApplicationContext applicationContext;

    @Resource
    private ExecutorService executorService;

    @Resource(name = "stringRedisTemplate")
    private StringRedisTemplate redisTemplate;

    /** 操作员姓名,派件、揽件、接单 */
    private static final String DELIVER_NAME= "deliverName";

    /** 操作员联系方式 */
    private static final String CONTACT_INFO = "contactInfo";

    @Override
    public Pagenation<LogisticsOrderVO> queryList(QueryLogisticsOrderDto queryItem, Integer pageNum, Integer pageSize) {
        DubboResult<PagenationDto<LogisticsOrderDto>> result = remoteLogisticsOrderService.queryLogisticsOrderPage(queryItem,
                                                                                                                   pageNum,
                                                                                                                   pageSize);
        PagenationDto<LogisticsOrderDto> pageDto = result.getResult();
        List<LogisticsOrderDto> logisticsOrderDtoList = pageDto.getList();
        Pagenation<LogisticsOrderVO> page = new Pagenation<>();
        page.setTotal(pageDto.getTotal());
        List<LogisticsOrderVO> logisticsOrderVOList = transformLogisticsOrderVO(logisticsOrderDtoList);
        page.setList(logisticsOrderVOList);
        return page;
    }

    /**
     * LogisticsOrderDto转LogisticsOrderVO，用于页面显示
     *
     * @author chencheng
     * @param logisticsOrderDtoList
     * @return
     */
    private List<LogisticsOrderVO> transformLogisticsOrderVO(List<LogisticsOrderDto> logisticsOrderDtoList) {
        List<LogisticsOrderVO> logisticsOrderVOList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(logisticsOrderDtoList)) {
            return logisticsOrderVOList;
        }
        for (LogisticsOrderDto dto : logisticsOrderDtoList) {
            LogisticsOrderVO vo = BeanUtils.copy(dto, LogisticsOrderVO.class);
            vo.setGmtCreate(DateUtils.getSecondStr(dto.getGmtCreate()));
            vo.setGmtModified(DateUtils.getSecondStr(dto.getGmtModified()));
            LogisticsOrderStatusEnum statusEnum = LogisticsOrderStatusEnum.getByCode(dto.getOrderStatus());
            if (Objects.isNull(statusEnum)){
                vo.setOrderStatus(dto.getOrderStatus());
            }else {
                vo.setOrderStatus(statusEnum.getDesc());
            }
            vo.setOrderStatusModified(DateUtils.getSecondStr(dto.getOrderStatusModified()));
            logisticsOrderVOList.add(vo);
        }
        return logisticsOrderVOList;
    }

    @Override
    public Integer updateLogisticsOrderStatus(String logisticsId, String orderStatus) {
        DubboResult<Integer> result = remoteLogisticsOrderService.updateBylogisticsId(logisticsId, orderStatus);
        return result.getResult();
    }

    @Override
    public YTOLogisticsReturnVO ytoLogisticsMsg(String logisticsInterface, String dataDigest, String clientId,
                                                String type) {
        YTOLogisticsReturnVO ytoLogisticsReturnVO = new YTOLogisticsReturnVO(ExpressCompanyEnum.YTO.getCode());
        // 需要将被转义的字符串转义回来。不用在url解码
        String newLogisticsInterface = StringEscapeUtils.unescapeHtml3(logisticsInterface);
        // 物流信息xml转JavaBean
        AppLogUtil.info(LOG,newLogisticsInterface);   //打印查看物流信息
        QueryTYOLogisticsMsgDto dto = JaxbUtil.converyToJavaBean(newLogisticsInterface, QueryTYOLogisticsMsgDto.class);

        if (StringUtils.isBlank(dto.getMailNo())) {
            ytoLogisticsReturnVO.setSuccessAndReason(YTOErrorCodeReturnEnum.LOGISTICS_ERROR);
            return ytoLogisticsReturnVO;
        }
        ytoLogisticsReturnVO.setTxLogisticID(dto.getTxLogisticID());
        // 校验信息
        checkytoLogisticsMsgParam(ytoLogisticsReturnVO, newLogisticsInterface, dataDigest, clientId);

        if (!ytoLogisticsReturnVO.getSuccess()) {
            return ytoLogisticsReturnVO;
        }
        // JavaBean转logisticsOrderDto，用于请求修改物流状态
        LogisticsOrderDto logisticsOrderDto = new LogisticsOrderDto();
        YTOLogisticsStatusEnum ytoLogisticsStatusEnum = YTOLogisticsStatusEnum.getByCode(dto.getInfoContent());
        if (StringUtils.isBlank(dto.getInfoContent()) || ytoLogisticsStatusEnum == null) {
            ytoLogisticsReturnVO.setSuccessAndReason(YTOErrorCodeReturnEnum.INFO_CONTENT_ERROR);
            return ytoLogisticsReturnVO;
        }

        logisticsOrderDto.setLogisticsStatus(ytoLogisticsStatusEnum.getDesc());
        logisticsOrderDto.setPostId(dto.getMailNo());
        logisticsOrderDto.setLogisticsCode(ExpressCompanyEnum.YTO.getCode());



        //派件状态，获取派件员信息
        if(ytoLogisticsStatusEnum == YTOLogisticsStatusEnum.SENT_SCAN){
            Map<String,String> deliveryMsg = new HashMap<>();
            deliveryMsg.put(DELIVER_NAME,dto.getDeliveryName());
            deliveryMsg.put(CONTACT_INFO,dto.getContactInfo());
            logisticsOrderDto.setDeliveryMsg(deliveryMsg);
            logisticsOrderDto.setOrderStatus(LogisticsOrderStatusEnum.SENT_SCAN.getCode());
        // 签收操作。需要更改物流子订单状态
        }else if (ytoLogisticsStatusEnum == YTOLogisticsStatusEnum.SIGNED) {
            String lastProcessInfo = getLastProcessInfo(LogisticsNameEnum.YTO.getCode(),dto.getMailNo());
            String[] refuseRemark = { "签收人: 退件", "签收人: 退回" };
            if(StringUtils.indexOfAny(lastProcessInfo, refuseRemark) > 0){
                logisticsOrderDto.setOrderStatus(LogisticsOrderStatusEnum.RETURN.getCode());
            }else{
                logisticsOrderDto.setOrderStatus(LogisticsOrderStatusEnum.SUCCESS.getCode());
            }
        //签收失败
        } else if (ytoLogisticsStatusEnum == YTOLogisticsStatusEnum.FAILED) {
            logisticsOrderDto.setOrderStatus(LogisticsOrderStatusEnum.FAILED.getCode());
        //接单
        } else if (ytoLogisticsStatusEnum == YTOLogisticsStatusEnum.ACCEPT){
            logisticsOrderDto.setOrderStatus(LogisticsOrderStatusEnum.ACCEPT.getCode());
        //已收件
        } else if (ytoLogisticsStatusEnum == YTOLogisticsStatusEnum.GOT){
            logisticsOrderDto.setOrderStatus(LogisticsOrderStatusEnum.UNDER_WAY.getCode());
        }
        // 修改物流状态
        DubboResult<Integer> result = remoteLogisticsOrderService.updateLogisticsOrderStatus(logisticsOrderDto);
        if (!result.isSuccess() || result.getResult() == null || result.getResult().intValue() <= 0) {
            ytoLogisticsReturnVO.setSuccessAndReason(YTOErrorCodeReturnEnum.TXLOGISTICID_NON_ERROR);
        }
        return ytoLogisticsReturnVO;
    }

    /**
     * 查询最近一条走件信息
     * 这边暂时先写死物流公司代码为-圆通
     * @param postId
     * @return
     */
    private String getLastProcessInfo(String logisticsCode,String postId) {
        try{
            DubboResult<List<LogisticsProcessDto>> result = remoteLogisticsOrderService.queryWaybillTrace(logisticsCode,postId);
            if(result.isSuccess() && CollectionUtils.isNotEmpty(result.getResult())){
                LogisticsProcessDto dto = result.getResult().get(0);
                return dto.getProcessInfo();
            }
            throw new QihoManagerException("没有查到走件信息，postId="+postId);
        }catch (Exception e){
            LOG.warn("queryWaybillTrace接口查询超时：logisticsCode:{},postId:{}"+logisticsCode,postId);
            throw new QihoManagerException("没有查到走件信息，postId="+postId);
        }
    }

    /**
     * 校验接口元素
     *
     * @author chencheng
     * @param ytoLogisticsReturnVO
     * @param logisticsInterface
     * @param dataDigest
     * @param clientId
     * @return
     */
    public void checkytoLogisticsMsgParam(YTOLogisticsReturnVO ytoLogisticsReturnVO, String logisticsInterface,
                                          String dataDigest, String clientId) {
        // 校验客户编码
        if (!YTOInterfaceUtil.getClientId().equals(clientId)) {
            ytoLogisticsReturnVO.setSuccessAndReason(YTOErrorCodeReturnEnum.CLIENTID_ERROR);
            return;
        }

        // 校验加密
        String encryptionSign = MessageDigestUtils.toMD5Base64(logisticsInterface + YTOInterfaceUtil.getPartnerId());
        if (!StringUtils.equalsIgnoreCase(dataDigest, encryptionSign)) {
            ytoLogisticsReturnVO.setSuccessAndReason(YTOErrorCodeReturnEnum.SIGN_ERROR);
            return;
        }
        ytoLogisticsReturnVO.setSuccessAndReason(YTOErrorCodeReturnEnum.SUCCESS);
    }

    @Override
    public List<LogisticsProcessDto> queryYTOLogisticsProcessByPostId(String postId,String logisticsCode) {
        DubboResult<List<LogisticsProcessDto>> result = remoteLogisticsOrderService.queryWaybillTrace(logisticsCode,postId);
        return result.getResult();
    }

    @Override
    public OrderDealWithVO batchSign(BatchAuditOrderParam param, String status) {
        return orderDealWithService.submitDealWithTask(param.getFileUrl(), status, LogisticsOrderSignHandler.class);
    }

    @Override
    public Boolean manualErpLogisticsSyncQueryTaskJob() {
        if (Environment.isPrepub() || Environment.isOnline()) {
            LOG.info("预发环境和线上环境不执行手动触发ERP物流查询任务");
            return false;
        }
        DubboResult<Boolean> result = remoteLogisticsOrderService.manualErpLogisticsSyncQueryJob();
        return result.getResult();
    }

    /**
     * @see com.qiho.manager.biz.service.LogisticsOrderService#syncLogisticsStatus(String)
     */
    @Override
    public void syncLogisticsStatus(String logisticsOrderStatus) {
        LOG.warn("开始同步物流状态");
        remoteLogisticsOrderService.syncLogisticsStatus(logisticsOrderStatus);
    }

    @Override
    public Boolean updateLogisticsOrderRemark(LogisticsOrderDto param) {
        return remoteLogisticsOrderService.updateLogisticsOrderRemark(param);
    }


    @Override
    public Map<String, Object> submitExportDistributeOrder() {
        int count = remoteLogisticsOrderService.queryDistributeOverTimeNum();
        if(count <= 0 ){
            throw new QihoManagerException("没有需要导出的记录");
        }
        DistributeOrderExportRunnable runnable = applicationContext.getBean(DistributeOrderExportRunnable.class);
        Map<String,Object> info = runnable.init();
        info.put("count",count);
        executorService.submit(runnable);
        return info;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
    }

    @Override
    public JSONObject queryExpressBlackExportStatus(String taskId) {
        CacheConstantseEnum constantseEnum = CacheConstantseEnum.EXPORT_DISTRIBUTE_LOGISTICS_ORDER;
        String cacheKey = constantseEnum.getCacheKey(taskId);
        BoundHashOperations<String,String,Object> operations = redisTemplate.boundHashOps(cacheKey);
        Map<String,Object> info = operations.entries();
        JSONObject json = new JSONObject();
        if(info.isEmpty()){
            json.put("success",true);
        }else{
            json.putAll(info);
        }
        json.put("success",json.getBoolean("success"));
        return json;
    }

    @Override
    public Pagenation<LogisticsOrderVO> queryServiceList(QueryLogisticsOrderDto queryItem, Integer pageNum, Integer pageSize) {
        PagenationDto<LogisticsOrderDto> result = remoteLogisticsOrderService.queryServiceLogisticsOrder(queryItem,
                pageNum,
                pageSize);
        List<LogisticsOrderDto> logisticsOrderDtoList = result.getList();
        Pagenation<LogisticsOrderVO> page = new Pagenation<>();
        page.setTotal(result.getTotal());
        List<LogisticsOrderVO> logisticsOrderVOList = transformLogisticsOrderVO(logisticsOrderDtoList);
        page.setList(logisticsOrderVOList);
        return page;
    }
}
