package com.qiho.manager.biz.service.impl;

import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.collect.Lists;
import com.qiho.center.api.dto.PopupWindowDto;
import com.qiho.center.api.enums.PopupWindowDeletedFailEnum;
import com.qiho.center.api.remoteservice.RemotePopupWindowService;
import com.qiho.manager.biz.service.PopupWindowService;
import com.qiho.manager.biz.vo.PopupWindowVO;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author jiafeng.geng
 * @date 2018-05-30
 */
@Service
public class PopupWindowServiceImpl implements PopupWindowService {

    private static final Logger LOGGER = LoggerFactory.getLogger(PopupWindowServiceImpl.class);

    @Resource
    private RemotePopupWindowService remotePopupWindowService;

    @Override
    public Integer add(PopupWindowDto popupWindowDto) {
        return remotePopupWindowService.add(popupWindowDto);
    }

    @Override
    public Integer deleteList(List<Long> ids) {
        // 判断弹窗id集合是否为空
        if (CollectionUtils.isEmpty(ids)) {
            LOGGER.warn("弹窗id集合为空：ids={}", ids);
            return PopupWindowDeletedFailEnum.IDS_IS_NULL.getCode();
        }

        // 根据弹窗id集合查询弹窗对象集合
        List<PopupWindowDto> popupWindowDtoList = remotePopupWindowService.queryByIds(ids).getList();
        if (CollectionUtils.isEmpty(popupWindowDtoList)) {
            LOGGER.warn("根据弹窗id未查询到任何结果：ids={}", ids);
            return PopupWindowDeletedFailEnum.DATA_IS_NULL.getCode();
        }

        // 获取所有弹窗状态为关闭的弹窗id
        List<Long> idList = popupWindowDtoList.stream().filter(e -> Boolean.FALSE.equals(e.getPopupWindowStatus()))
                .map(PopupWindowDto::getId)
                .collect(Collectors.toList());

        // 查询到的弹窗数量和状态为关闭的弹窗数量一致，说明要删除的弹窗中没有开启状态的
        // 反之则说明要删除的弹窗中包含开启状态的弹窗
        if (popupWindowDtoList.size() != idList.size()) {
            LOGGER.warn("要删除的弹窗列表中包含开启状态的弹窗：{}", popupWindowDtoList);
            return PopupWindowDeletedFailEnum.POPUP_WINDOW_STATUS_ON.getCode();
        }

        // 根据弹窗id删除所有弹窗状态为关闭的弹窗
        return remotePopupWindowService.deleteList(idList);
    }

    @Override
    public Integer update(PopupWindowDto popupWindowDto) {
        // 判断弹窗状态是否为开启
        if (popupWindowDto.getPopupWindowStatus()) {
            // 关闭已开启的弹窗状态
            remotePopupWindowService.closeOnStatus();
        }
        return remotePopupWindowService.update(popupWindowDto);
    }

    @Override
    public List<PopupWindowVO> queryList() {
        // 查询所有弹窗对象集合
        List<PopupWindowDto> popupWindowDtoList = remotePopupWindowService.queryList().getList();

        // 类型转换
        List<PopupWindowVO> popupWindowVOList = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(popupWindowDtoList)) {
            popupWindowDtoList.forEach(e -> popupWindowVOList.add(BeanUtils.copy(e, PopupWindowVO.class)));
        }
        return popupWindowVOList;
    }
}
