package com.qiho.manager.biz.process.component;

import cn.com.duiba.wolf.utils.SecurityUtils;
import com.google.common.collect.Sets;
import com.google.common.io.CharSink;
import com.google.common.io.FileWriteMode;
import com.google.common.io.Files;
import com.qiho.center.api.dto.component.ComponentDto;
import com.qiho.center.api.enums.component.ComponentTypeEnum;
import com.qiho.center.api.enums.page.PageTypeEnum;
import com.qiho.manager.biz.params.component.ComponentPagePreParam;
import com.qiho.manager.biz.vo.component.ComponentCodeVO;
import com.qiho.manager.biz.vo.component.ComponentPagePreviewVO;
import com.qiho.manager.common.constant.DomainConstantUtil;
import com.qiho.manager.common.exception.QihoManagerException;
import com.qiho.manager.common.util.AssertUtil;
import com.qiho.manager.common.util.FileIDGenerator;
import com.qiho.manager.common.util.UploadTool;
import org.apache.commons.lang.StringUtils;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.ui.velocity.VelocityEngineUtils;

import java.io.File;
import java.nio.charset.Charset;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 集合页组件化页面处理器
 *
 * @author chensong
 * @create 2018-09-20
 */
@Component
public class CollectionComponentPageProcessor extends ComponentPageProcessor implements InitializingBean {

    private static final Logger LOGGER = LoggerFactory.getLogger(CollectionComponentPageProcessor.class);

    @Value("${component.page.collection.vendor.js}")
    private String vendorCollectionJsUrl;

    @Value("${component.page.base.css}")
    private String baseCssUrl;

    @Value("${component.page.collection.main.js}")
    private String mainCollectionJsUrl;


    @Override
    public void afterPropertiesSet() throws Exception {
        ComponentPageDecider.registerPorcessor(PageTypeEnum.COMPONENT_COLLECTION_PAGE, this);
    }

    @Override
    public ComponentPagePreviewVO preview(ComponentPagePreParam param) {

        List<Long> idList = param.fetchComponentIds();

        // 参数校验
        List<ComponentDto> componentDtoList = checkPreParam(param, idList);

        //代码打包 获得渲染后HTML代码的md5 和Html代码文件的url
        Map<String, String> htmlMap = packageCode(componentDtoList, param, idList);

        // 组装预览url返回给前端
        ComponentPagePreviewVO componentPre = new ComponentPagePreviewVO();
        componentPre.setMd5(htmlMap.get(HTML_MD5_KEY));

        String htmlFileUrl = htmlMap.get(HTML_URL_KEY);
        componentPre.setPageFileUrl(htmlFileUrl);

        String prePath = DomainConstantUtil.getQihoWebUrl() + "/page/preview?id="
                + param.getId() +"&fileUrl=" + htmlFileUrl ;
        componentPre.setPreUrl(prePath);

        return componentPre;
    }


    /**
     *  页面预览参数校验
     *  校验不通过时throw QihoManagerException
     *
     *  校验项
     *      -> 前端传的组件List不能为空
     *      -> 每个组件的有效性
     *      -> banner、浮标组件最多一个
     *
     * @param param  组件预览的参数
     * @param idList 预览参数中组件id集合
     * @return 组件具体信息列表
     */
    private List<ComponentDto> checkPreParam(ComponentPagePreParam param, List<Long> idList){

        // 组件列表校验
        AssertUtil.collectionNotEmpty(param.getList(), "组件列表不能为空");

        // 校验组件有效数量
        List<ComponentDto> componentDtoList = remoteComponentService.listBatchByIds(idList);
        Set<Long> idSet = Sets.newHashSet(idList);
        if (idSet.size() != componentDtoList.size()) {
            idSet.removeAll(componentDtoList.stream().map(ComponentDto::getId).collect(Collectors.toSet()));
            throw new QihoManagerException("组件id = " + idSet.toString() + "不存在");
        }

        // 校验各组件类型的数量
        EnumMap<ComponentTypeEnum, Integer> componentTypeMap = new EnumMap<>(ComponentTypeEnum.class);
        componentDtoList.forEach(e -> {
            if (componentTypeMap.containsKey(e.getComponentType())) {
                componentTypeMap.put(e.getComponentType(), componentTypeMap.get(e.getComponentType()) + 1 );
            } else {
                componentTypeMap.put(e.getComponentType(), 1);
            }
        });

        if (componentTypeMap.get(ComponentTypeEnum.BANNER) != null && componentTypeMap.get(ComponentTypeEnum.BANNER) > 1) {
            throw new QihoManagerException("保存失败，请检查组件配置是否正确");
        }
        if (componentTypeMap.get(ComponentTypeEnum.COLLECTION_BUOY) != null && componentTypeMap.get(ComponentTypeEnum.COLLECTION_BUOY) > 1) {
            throw new QihoManagerException("保存失败，请检查组件配置是否正确");
        }

        return componentDtoList;
    }


    /**
     * 获取从oss获取到各个组件的代码  拆分代码
     * 分别打包css，js代码成文件，上传到CDN，获取到文件url
     * 替换pageShell.vm的占位符，生成HTML文件 上传到OSS
     *
     * @param componentDtoList
     * @return html文件url html代码md5
     */
    private Map<String, String> packageCode(List<ComponentDto> componentDtoList, ComponentPagePreParam param, List<Long> idList){

        // 从Oss下载代码
        Map<Long, ComponentCodeVO> componentCodeMap = downloadCode(componentDtoList);

        // 打包css js文件上传Oss 获得css文件url js文件url html代码
        Map<String, String> codeFileMap = buildCodeFile(componentCodeMap, idList);

        // CollectionPageShell.vm模板占位符 将模板代码上传至Oss
        return renderTemplate(codeFileMap, param);
    }


    /**
     * 渲染模板 获取HTML代码上传到oss
     * @param codeFileMap
     * @param param
     * @return HTML文件地址 HTML代码的md5
     */
    private Map<String, String> renderTemplate(Map<String, String> codeFileMap, ComponentPagePreParam param){
        if (StringUtils.isBlank(codeFileMap.get(CSS_URL_KEY))
                || StringUtils.isBlank(codeFileMap.get(JS_URL_KEY))) {
            LOGGER.error("获取代码失败，codeFileMap = {}", codeFileMap);
            throw new QihoManagerException("代码获取失败");
        }

        Map<String, Object> model = new HashMap<>(16);
        model.put("jsReplace", codeFileMap.get(JS_URL_KEY));
        model.put("cssReplace", codeFileMap.get(CSS_URL_KEY));
        model.put("htmlReplace", codeFileMap.get(HTML_CODE_KEY));
        model.put("jsConfigReplace", param.getTotalConfig());
        model.put("baseCssReplace", baseCssUrl);
        model.put("vendorJsReplace", vendorCollectionJsUrl);
        model.put("mainJsReplace", mainCollectionJsUrl);

        Map<String, String> htmlMap = new HashMap<>(16);

        // 渲染模板 输出HTML内容
        String context = VelocityEngineUtils.mergeTemplateIntoString(velocityEngine, "CollectionPageShell.vm", "UTF-8", model);

        String md5 = SecurityUtils.encode2StringByMd5(context);
        htmlMap.put(HTML_MD5_KEY, md5);

        // HTML代码上传至Oss
        String htmlFileName = FileIDGenerator.getRandomString(10);
        try {
            File htmlFile = File.createTempFile(htmlFileName,".html");
            CharSink charSink = Files.asCharSink(htmlFile, Charset.defaultCharset(), FileWriteMode.APPEND);
            charSink.write(context);

            String htmlFilePath = PATH_PRE + new DateTime().toString(DATETIME_FORMAT) + URL_SEPARATOR + htmlFile.getName();
            boolean jsResult = UploadTool.uploadOssHtml(htmlFile, htmlFilePath);
            if (!jsResult) {
                throw new QihoManagerException("上传Js文件失败");
            }

            htmlMap.put(HTML_URL_KEY, HTTP_URL_PRE + ossYunUrl + htmlFilePath);

        } catch (Exception e) {
            LOGGER.error("上传文件异常", e);
            throw new QihoManagerException(e.getMessage());
        }

        return htmlMap;
    }


}
