package com.qiho.manager.biz.runnable;

import cn.afterturn.easypoi.excel.ExcelExportUtil;
import cn.afterturn.easypoi.excel.entity.ExportParams;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.google.common.collect.Lists;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.bklist.BaiqiBlackListDto;
import com.qiho.center.api.params.blacklist.BaiqiBlackListQueryParam;
import com.qiho.center.api.remoteservice.blackList.RemoteBaiqiBlackListService;
import com.qiho.center.api.remoteservice.blackList.RemoteBlackListService;
import com.qiho.manager.biz.params.BlackExportParam;
import com.qiho.manager.common.constant.CacheConstantseEnum;
import com.qiho.manager.common.util.AppLogUtil;
import com.qiho.manager.common.util.UploadTool;
import org.apache.poi.ss.usermodel.Workbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.data.redis.core.HashOperations;
import org.springframework.stereotype.Component;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.Map;

/**
 * 
 * ClassName: BlackListExportRunnable <br/>
 * Function: 黑名单导出. <br/>
 * date: 2017年6月27日 下午3:40:34 <br/>
 *
 * @author chencheng
 * @version
 */
@Component
@Scope("prototype")
public class BlackListExportRunnable extends BaseExportRunnable implements Runnable {

    private static final Logger LOGGER = LoggerFactory.getLogger(BlackListExportRunnable.class);

    private static final int       ORDER_EXPORT_MAX = 5000;

    @Autowired
    private RemoteBlackListService remoteBlackListService;

    private BaiqiBlackListQueryParam        params;

    private String                 filePath         = "blackList/";

    private File                   file;

    private Long taskId = System.currentTimeMillis();

    @Autowired
    private RemoteBaiqiBlackListService remoteBaiqiBlackListService;

    /**
     * 初始化
     * 
     * @return 任务号
     */
    public Map<String, Object> init() {
        file = super.getFile2(filePath);
        return super.init(file,taskId);
    }

    @Override
    public void run() {
        int exportCount = remoteBaiqiBlackListService.queryBlackListPageCount(params);

        int batch = getBatchs(exportCount);
        CacheConstantseEnum constantseEnum = CacheConstantseEnum.EXPORT_ORDER;
        String key = constantseEnum.getCacheKey(taskId);
        HashOperations<String, String, Object> operations = redisTemplate.opsForHash();
        Workbook workbook = null;
        try (FileOutputStream outputStream = new FileOutputStream(file)){
            List<BlackExportParam> exportList = Lists.newArrayList();
            for (int i = 0; i < batch; i++) {
                params.setOffset(i * ORDER_EXPORT_MAX);
                PagenationDto<BaiqiBlackListDto> result = remoteBaiqiBlackListService.queryBlackListPage(params);
                List<BlackExportParam> list = transferExcelEntity(result);
                exportList.addAll(list);

                operations.increment(key, "successCount", list.size());
            }
            workbook = ExcelExportUtil.exportBigExcel(new ExportParams(),BlackExportParam.class,exportList);
            workbook.write(outputStream);
            operations.put(key, "message", "文件上传中");
            String url = UploadTool.uploadOssNotCDN(file, filePath + file.getName(),
                                                    "application/x-msdownload;charset=GBK");

            operations.put(key, "url", url);
            operations.put(key, "message", "导出成功");
        } catch (Exception e) {
            AppLogUtil.error(LOGGER,"black Export error msg={}",e);
            operations.put(key, "message", e.getMessage());
        } finally {
            operations.put(key, "success", "true");
            ExcelExportUtil.closeExportBigExcel();
            try {
                workbook.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
            if (file != null && file.exists() && !file.delete()) {
                AppLogUtil.warn(LOGGER,"file delete error or file not exists");
            }
        }
    }
    /**
     * 转换成Excel导出的格式
     * @param result
     * @return
     */
    private List<BlackExportParam> transferExcelEntity(PagenationDto<BaiqiBlackListDto> result){

        List<BlackExportParam> list = Lists.newArrayList();
        for (BaiqiBlackListDto dto : result.getList()) {
            BlackExportParam ordersExport = BeanUtils.copy(dto, BlackExportParam.class);
            ordersExport.setDealType(dto.getSourceType().getDesc());
            list.add(ordersExport);
        }

        return list;
    }


    private int getBatchs(int total) {
        return total % ORDER_EXPORT_MAX == 0 ? total / ORDER_EXPORT_MAX : (total / ORDER_EXPORT_MAX + 1);
    }

    public BaiqiBlackListQueryParam getParams() {
        return params;
    }

    public void setParams(BaiqiBlackListQueryParam params) {
        params.setMax(ORDER_EXPORT_MAX);
        params.setOffset(0);
        this.params = params;
    }
}
