/**
 * Project Name:qiho-manager-biz File Name:OrderServiceImpl.java Package Name:com.qiho.manager.biz.service.impl
 * Date:2017年6月6日下午2:26:13 Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 */

package com.qiho.manager.biz.service.impl;

import cn.com.duiba.sso.api.tool.RequestTool;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.DateUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.dto.*;
import com.qiho.center.api.dto.logistics.LogisticsOrderDto;
import com.qiho.center.api.dto.order.BatchOrderDto;
import com.qiho.center.api.dto.order.BatchOrderExtDto;
import com.qiho.center.api.dto.order.BatchOrderProcessDto;
import com.qiho.center.api.dto.page.BaiqiPageDto;
import com.qiho.center.api.enums.*;
import com.qiho.center.api.params.OrderConsumerParam;
import com.qiho.center.api.params.OrderItemParam;
import com.qiho.center.api.params.OrderPageParams;
import com.qiho.center.api.remoteservice.RemoteConfigService;
import com.qiho.center.api.remoteservice.RemoteLogisticsOrderService;
import com.qiho.center.api.remoteservice.RemoteSkinService;
import com.qiho.center.api.remoteservice.order.*;
import com.qiho.center.api.remoteservice.page.RemotePageBackendService;
import com.qiho.manager.biz.params.BatchAuditOrderParam;
import com.qiho.manager.biz.runnable.OrdersExportRunnable;
import com.qiho.manager.biz.service.OrderDealWithService;
import com.qiho.manager.biz.service.OrderService;
import com.qiho.manager.biz.vo.*;
import com.qiho.manager.common.constant.CacheConstantseEnum;
import com.qiho.manager.common.exception.QihoManagerException;
import com.qiho.manager.common.util.Environment;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static com.qiho.center.api.constant.ConfigConstant.ORDER_PRIMITIVE_AUTH_EMAIL;
import static com.qiho.manager.common.util.StringUtil.hideConsumerInfo;
import static com.qiho.manager.common.util.StringUtil.hideMobileInfo;

/**
 * ClassName:OrderServiceImpl <br/>
 * Date: 2017年6月6日 下午2:26:13 <br/>
 * 
 * @author zhanglihui
 * @version
 * @since JDK 1.6
 * @see
 */
@Service
public class OrderServiceImpl implements OrderService,ApplicationContextAware {

    private static final Logger logger = LoggerFactory.getLogger(OrderServiceImpl.class);

    @Autowired
    private RemoteOrderService remoteOrderService;

    @Autowired
    private RemoteFundOrderService remoteFundOrderService;

    @Autowired
    private RemoteErpOrderService remoteErpOrderService;

    @Autowired
    private RemoteLogisticsOrderService remoteLogisticsOrderService;

    @Autowired
    private RemoteChannelInfoService remoteChannelInfoService;

    @Autowired
    private RemoteAnticheateService remoteAnticheateService;

    @Autowired
    private RemoteAfterSaleOrderService remoteAfterSaleOrderService;


    @Autowired
    private ApplicationContext applicationContext;

    @Resource
    private ExecutorService executorService;

    @Autowired
    private OrderDealWithService orderDealWithService;

    @Autowired
    private RemoteSkinService remoteSkinService;

    @Resource
    private RemotePageBackendService  remotePageBackendService;

    @Resource(name = "redisTemplate")
    private AdvancedCacheClient advancedCacheClient;

    @Autowired
    private RemoteConfigService remoteConfigService;

    @Override
    public OrderDetailVO findByOrderId(String orderId) {
        OrderDetailVO orderDetail = new OrderDetailVO();
        OrderDto orderDto = remoteOrderService.findByOrderId(orderId).getResult();
        // 获取订单基本信息
        OrderVO order = getOrderVO(orderDto);
        orderDetail.setOrder(order);
        // 获取订单商品详情
        OrderItemVO orderItemVO = BeanUtils.copy(orderDto.getOrderItem(), OrderItemVO.class);
        orderDetail.setOrderItem(orderItemVO);
        // 获取订单资金流水
        List<FundVO> fundList = getFundList(orderId);
        orderDetail.setFundList(fundList);
        // 获取ERP信息
        List<ErpOrderVO> erpOrderList = getErpOrderList(orderId);
        orderDetail.setErpOrderList(erpOrderList);
        // 获取物流信息
        List<LogisticsVO> logisticsList = getLogisticsList(orderId);
        orderDetail.setLogisticsList(logisticsList);
        // 获取渠道信息
        ChannelInfoVO channelInfo = getChannelInfoVO(orderId);
        orderDetail.setChannelInfo(channelInfo);
        // 获取防作弊命中规则
        List<String> anticheateList = getAnticheateList(orderId);
        orderDetail.setAnticheateList(anticheateList);
        // 获取皮肤信息
        OrderSkinVO skinItem = getSkinVO(channelInfo.getUrl(), channelInfo.getSkinId());
        orderDetail.setSkinItem(skinItem);
        // 获取售后子订单
        List<AfterSaleVO> afterSaleList = getAfterSaleList(orderId);
        orderDetail.setAfterSaleList(afterSaleList);

        // 获取优惠券信息
        // update note：下线优惠券模块 优惠券查询返回空List
        orderDetail.setCouponList(new ArrayList<>());

        // 处理订单扩展字段
        setOrderExt(orderId, orderDetail, orderDto);

        return orderDetail;
    }



    private List<AfterSaleVO> getAfterSaleList(String orderId){
        List<AfterSaleOrderDto> dtoList = remoteAfterSaleOrderService.findAll(orderId);
        List<AfterSaleVO> afterSaleList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(dtoList)){
            return afterSaleList;
        }
        for (AfterSaleOrderDto dto : dtoList){
            AfterSaleVO afterSale = BeanUtils.copy(dto,AfterSaleVO.class);
            afterSale.setAfterSaleType(AfterSaleOrderTypeEnum.getByCode(dto.getAfterSaleType()).getDesc());
            afterSale.setAfterSaleReason(AfterSaleOrderReasonEnum.getByCode(dto.getAfterSaleReason()).getMsg());
            afterSale.setAfterSaleStatus(AfterSaleOrderStatusEnum.getByCode(dto.getAfterSaleStatus()).getMsg());
            afterSale.setGmtCreate(DateUtils.getSecondStr(dto.getGmtCreate()));
            afterSale.setGmtModified(DateUtils.getSecondStr(dto.getGmtModified()));
            afterSale.setFreightAmt(dto.getRefreightAmt());
            afterSale.setAfterSaleId(dto.getAfterSaleId());
            afterSaleList.add(afterSale);
        }
        return afterSaleList;
    }

    private OrderSkinVO getSkinVO(String url, Long skinId) {

        OrderSkinVO skinItem = new OrderSkinVO();

        skinItem.setSkinId(skinId);

        //新页面
        if(url.contains("detail2")){

            BaiqiPageDto pageDto = remotePageBackendService.findById(skinId);
            if(pageDto != null){
                skinItem.setSkinName(pageDto.getPageName());
            }
        }else {
            // 老皮肤
            try {
                DubboResult<SkinDto> result = remoteSkinService.querySkinById(skinId);
                if (result.isSuccess() && result.getResult() != null) {
                    skinItem.setSkinName(result.getResult().getSkinName());
                }
            } catch (QihoManagerException e) {
                logger.error("order detail get skin info error, skinId:{} ", skinId);
            }
        }

        return skinItem;
    }

    @Override
    public Map<String,Object> submitOrdersExport(OrderPageParams params) {
        OrdersExportRunnable runnable = applicationContext.getBean(OrdersExportRunnable.class);
        runnable.setParams(params);
        Map<String,Object> info = runnable.init();
        executorService.submit(runnable);
        return info;
    }
    
    /**
     * @author zhanglihui
     * @param orderId
     * @return
     */
    private List<String> getAnticheateList(String orderId) {
        AnticheateDto anticheateDto = remoteAnticheateService.findByOrderId(orderId).getResult();
        return anticheateDto.getRules();
    }

    /**
     * @author zhanglihui
     * @param orderId
     * @return
     */
    private ChannelInfoVO getChannelInfoVO(String orderId) {
        ChannelInfoDto channelInfo = remoteChannelInfoService.findByOrderId(orderId).getResult();
        ChannelInfoVO channelInfoVO = BeanUtils.copy(channelInfo, ChannelInfoVO.class);
        channelInfoVO.setTongdunId(channelInfo.getTongdunId());
        channelInfoVO.setMaxentId(channelInfo.getMaxentId());

        return channelInfoVO;
    }

    /**
     * @author zhanglihui
     * @param orderId
     * @return
     */
    private List<LogisticsVO> getLogisticsList(String orderId) {
        List<LogisticsOrderDto> list = remoteLogisticsOrderService.findByOrderId(orderId).getResult();
        List<LogisticsVO> logisticsList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(list)) {
            return logisticsList;
        }
        for (LogisticsOrderDto logisticsOrder : list) {
            LogisticsVO logisticsVO = BeanUtils.copy(logisticsOrder, LogisticsVO.class);
            logisticsVO.setGmtCreate(DateUtils.getSecondStr(logisticsOrder.getGmtCreate()));
            logisticsVO.setGmtModified(DateUtils.getSecondStr(logisticsOrder.getGmtModified()));
            LogisticsOrderStatusEnum orderStatus = LogisticsOrderStatusEnum.getByCode(logisticsOrder.getOrderStatus());
            if(orderStatus != null){
                logisticsVO.setOrderStatus(orderStatus.getDesc());
            }
            logisticsVO.setOrderStatusModified(DateUtils.getSecondStr(logisticsOrder.getOrderStatusModified()));
            logisticsList.add(logisticsVO);
        }
        return logisticsList;
    }

    /**
     * @author zhanglihui
     * @param orderId
     * @return
     */
    private List<ErpOrderVO> getErpOrderList(String orderId) {
        List<ErpOrderDto> list = remoteErpOrderService.findByOrderId(orderId).getResult();
        List<ErpOrderVO> erpOrderList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(list)) {
            return erpOrderList;
        }
        for (ErpOrderDto erpOrder : list) {
            ErpOrderVO erpOrderVO = BeanUtils.copy(erpOrder, ErpOrderVO.class);
            erpOrderVO.setErpStatus(ErpStatusEnum.getByCode(erpOrder.getErpStatus()).getDesc());
            erpOrderVO.setErpType(ErpTypeEnum.getByCode(erpOrder.getErpType()).getDesc());
            erpOrderVO.setGmtCreate(DateUtils.getSecondStr(erpOrder.getGmtCreate()));
            erpOrderVO.setGmtModified(DateUtils.getSecondStr(erpOrder.getGmtModified()));
            erpOrderList.add(erpOrderVO);
        }
        return erpOrderList;
    }

    /**
     * @author zhanglihui
     * @param orderId
     * @return
     */
    private List<FundVO> getFundList(String orderId) {
        List<FundOrderDto> list = remoteFundOrderService.findByOrderId(orderId).getResult();
        List<FundVO> fundList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(list)) {
            return fundList;
        }
        for (FundOrderDto fundOrder : list) {
            FundVO fundVO = BeanUtils.copy(fundOrder, FundVO.class);
            fundVO.setFundStatus(FundStatusEnum.getEnumByCode(fundOrder.getFundStatus()).getDesc());
            fundVO.setPayType(PayTypeEnum.getEnumByCode(fundOrder.getPayType()).getDesc());
            fundVO.setGmtCreate(DateUtils.getSecondStr(fundOrder.getGmtCreate()));
            fundVO.setGmtModified(DateUtils.getSecondStr(fundOrder.getGmtModified()));
            fundVO.setBizType(FundBizTypeEnum.getByCode(fundOrder.getBizType()).getDesc());
            fundVO.setAfterSaleId(fundOrder.getAftersaleId());
            fundList.add(fundVO);
        }
        return fundList;
    }

    /**
     * @author zhanglihui
     * @return
     */
    private OrderVO getOrderVO(OrderDto orderDto) {
        OrderVO order = BeanUtils.copy(orderDto, OrderVO.class);

        // 订单敏感信息处理
        if (!orderPrimitiveAuth()) {
            order.setConsumerName(hideConsumerInfo(orderDto.getConsumerName()));
            order.setMobile(hideMobileInfo(orderDto.getMobile()));
            order.setIdCard(hideIdCard(orderDto.getIdCard()));
        }
        order.setGmtCreate(DateUtils.getSecondStr(orderDto.getGmtCreate()));
        order.setGmtModified(DateUtils.getSecondStr(orderDto.getGmtModified()));
        order.setDelivery(DeliveryEnum.getByCode(orderDto.getOrderItem().getDelivery()).getDesc());
        order.setOrderStatus(OrderStatusEnum.getByCode(orderDto.getOrderStatus()).getDesc());
        order.setSponsorAftersale(remoteAfterSaleOrderService.sponsorAfterSale(orderDto.getOrderId()));
        return order;
    }

    private void setOrderExt(String orderId, OrderDetailVO orderDetail, OrderDto orderDto) {
        String jsonExt = orderDto.getExtJson();
        if (StringUtils.isNotBlank(jsonExt)) {
            try {
                orderDetail.getOrder().setExt(JSONObject.parseObject(jsonExt, OrderDetailExtVo.class));
            } catch (Exception e) {
                logger.info("解析订单扩展数据错误 {}", orderId, e);
            }
        }
    }

    private  String hideIdCard(String idCard) {
        if(StringUtils.isBlank(idCard)){
            return idCard;
        }

        if(idCard.length() > 3 && idCard.length() < 7){
            return idCard.substring(0,3) + "****";
        }

        if(idCard.length() >= 7){
            return idCard.substring(0,3) + "****" + idCard.substring(idCard.length() -4 ,idCard.length());
        }
        return "****";
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext){
        this.applicationContext = applicationContext;
    }

    @Override
    public void updateOrderItem(OrderItemParam orderItemParam) {
        DubboResult<Void> result = remoteOrderService.updateOrderItem(orderItemParam);
        if(!result.isSuccess()){
            throw new QihoManagerException(result.getMsg());
        }
    }

    /**
     * @see com.qiho.manager.biz.service.OrderService#updateOrderConsumer(com.qiho.center.api.params.OrderConsumerParam)
     */
    @Override
    public void updateOrderConsumer(OrderConsumerParam orderConsumerParam) {
        DubboResult<Void> result = remoteOrderService.updateOrderConsumer(orderConsumerParam);
        if(!result.isSuccess()){
            throw new QihoManagerException(result.getMsg());
        }
    }
    
    @Override
    public OrderDealWithVO batchAuditOrder(BatchAuditOrderParam param) {
        //return orderDealWithService.submitDealWithTask(param.getFileUrl(), OrderAuditHandler.class);
        logger.info("BatchAuditOrder param:{},adminId:{}",JSON.toJSONString(param),RequestTool.getAdminId());
        BatchOrderDto batchOrderDto = new BatchOrderDto();
        batchOrderDto.setFileUrl(param.getFileUrl());
        //管理后台不涉及到具体的商家操作  这里-1做下兼容
        batchOrderDto.setMerchantId(-1L);
        String taskId = orderDealWithService.createTaskId();
        batchOrderDto.setTaskId(taskId);
        DubboResult<Integer> integerDubboResult = remoteOrderService.batchAuditOrder(batchOrderDto);
        OrderDealWithVO orderDealWithVO = new OrderDealWithVO();
        orderDealWithVO.setCount(integerDubboResult.getResult());
        orderDealWithVO.setTaskId(taskId);
        return orderDealWithVO;
    }

    @Override
    public Pagenation<OrderSnapshotVO> findApproveOrder(OrderPageParams params) {
        PagenationDto<OrderSnapshotDto> pagenationDto = remoteOrderService.findAllPage(params).getResult();
        List<OrderSnapshotDto> dtoList = pagenationDto.getList();
        if(CollectionUtils.isEmpty(dtoList)){
            return new Pagenation<>();
        }
        List<String> orderIddList = Lists.newArrayList();
        for (OrderSnapshotDto dto : dtoList){
            orderIddList.add(dto.getOrderId());
        }
        List<AnticheateDto> anticheateDtoList = remoteAnticheateService.findByOrderIds(orderIddList).getResult();
        for (int i=0;i<dtoList.size();i++){
            dtoList.get(i).setAnticheatRules(anticheateDtoList.get(i).getRules().toString());
        }
        List<OrderSnapshotVO> orderSnapshotVOList = Lists.transform(dtoList,dto ->{
            OrderSnapshotVO vo = BeanUtils.copy(dto,OrderSnapshotVO.class);
            vo.setGmtCreate(DateUtils.getSecondStr(dto.getGmtCreate()));
            vo.setOrderStatus(dto.getOrderStatusEnum().getDesc());
            vo.setFundStatus(dto.getFundStatusEnum()==null?"":dto.getFundStatusEnum().getDesc());
            vo.setAmt(dto.getOrderAmt());
            return vo;
        });
        Pagenation<OrderSnapshotVO> pageList = new Pagenation<>();
        pageList.setList(orderSnapshotVOList);
        pageList.setTotal(pagenationDto.getTotal());
        return pageList;
    }

    @Override
    public String overTask(String taskId) {
        // 添加Redis缓存, 缓存时间1小时
        String redisKey = CacheConstantseEnum.EXPORT_ORDER_CACEL + ":" + taskId;
        advancedCacheClient.set(redisKey, taskId, CacheConstantseEnum.EXPORT_ORDER_CACEL.getTime(),
                CacheConstantseEnum.EXPORT_ORDER_CACEL.getTimeUnit());
        return "任务结束成功：" + taskId;
    }

    @Override
    public boolean orderPrimitiveAuth() {
        try {
            List<String> authValue = advancedCacheClient.get(CacheConstantseEnum.ORDER_PRIMITIVE_AUTH_EMAIL.toString());
            if (authValue == null) {
                String configValue = remoteConfigService.fetchConfigByName(ORDER_PRIMITIVE_AUTH_EMAIL).getConfigValue();
                Splitter splitter = Splitter.on(",");
                authValue = splitter.splitToList(configValue);

                if (Environment.isDaily()) {
                    // 开发测试环境保留5分钟有效
                    advancedCacheClient.set(CacheConstantseEnum.ORDER_PRIMITIVE_AUTH_EMAIL.toString(), authValue,
                            5, TimeUnit.MINUTES);
                } else {
                    advancedCacheClient.set(CacheConstantseEnum.ORDER_PRIMITIVE_AUTH_EMAIL.toString(), authValue,
                            CacheConstantseEnum.ORDER_PRIMITIVE_AUTH_EMAIL.getTime(), CacheConstantseEnum.ORDER_PRIMITIVE_AUTH_EMAIL.getTimeUnit());
                }
            }
            String user = RequestTool.getAdmin().getEmail();
            return authValue.contains(user);
        } catch (Exception e){
            logger.error("查询订单原始信息权限错误, user={}", RequestTool.getAdmin().getName(), e);
            return false;
        }
    }

    @Override
    public void batchProcessOrder(Integer type, String fileUrl,String fileName) {
        BatchOrderProcessDto batchOrderProcessDto = new BatchOrderProcessDto();
        batchOrderProcessDto.setType(type);
        batchOrderProcessDto.setFileUrl(fileUrl);
        batchOrderProcessDto.setMerchantId(-1L);
        batchOrderProcessDto.setFileName(fileName);
        remoteOrderService.batchProcessOrder(batchOrderProcessDto);
    }

    /**
     * 获取订单号的导出状态
     * @param extList
     * @return
     */
    @Override
    public Map<String,Integer> getOrderExportStatusMap(final List<String> extList){
        if(CollectionUtils.isEmpty(extList)){
            return Maps.newHashMap();
        }
        final List<BatchOrderExtDto> orderExtDtos = Lists.newArrayList();
        extList.forEach(orderId->{
            final BatchOrderExtDto dto = new BatchOrderExtDto();
            dto.setOrderId(orderId);
            orderExtDtos.add(dto);
        });
        final List<BatchOrderExtDto> dtos = remoteOrderService.getOrderExt(orderExtDtos);
        if(CollectionUtils.isEmpty(dtos)){
            return Maps.newHashMap();
        }
        return dtos.stream().collect(Collectors.toMap(BatchOrderExtDto::getOrderId,
                BatchOrderExtDto::getOrderExportStatus,(a,b)->a));
    }
}
