package com.qiho.manager.biz.service.impl.component;

import cn.com.duiba.sso.api.tool.RequestTool;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.qiho.center.api.constant.PageInfoJsonKeyConstant;
import com.qiho.center.api.dto.ResultDto;
import com.qiho.center.api.dto.component.ComponentDto;
import com.qiho.center.api.dto.page.BaiqiPageDto;
import com.qiho.center.api.dto.page.PageComponentDetailDto;
import com.qiho.center.api.dto.page.PageComponentDto;
import com.qiho.center.api.dto.page.PageFormStrategyDto;
import com.qiho.center.api.enums.page.JSPlatformEnum;
import com.qiho.center.api.enums.page.PageStatusEnum;
import com.qiho.center.api.enums.page.PageTypeEnum;
import com.qiho.center.api.remoteservice.component.RemoteComponentService;
import com.qiho.center.api.remoteservice.page.RemotePageComponentService;
import com.qiho.manager.biz.params.component.ComponentPageCopyParam;
import com.qiho.manager.biz.params.component.ComponentPagePreParam;
import com.qiho.manager.biz.params.component.ComponentPageSaveParam;
import com.qiho.manager.biz.params.component.ComponentSimpleParam;
import com.qiho.manager.biz.process.component.ComponentPageDecider;
import com.qiho.manager.biz.process.component.ComponentPageProcessor;
import com.qiho.manager.biz.service.component.ComponentPageService;
import com.qiho.manager.biz.service.page.impl.PageServiceImpl;
import com.qiho.manager.biz.vo.component.ComponentPagePreviewVO;
import com.qiho.manager.biz.vo.component.ComponentSimpleVO;
import com.qiho.manager.biz.vo.page.PageComponentEditVO;
import com.qiho.manager.common.exception.QihoManagerException;
import com.qiho.manager.common.util.AssertUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 *
 * @author chensong
 * @create 2018-07-27
 */
@Service
@RefreshScope
public class ComponentPageServiceImpl implements ComponentPageService  {

    /** 日志 */
    private static final Logger LOGGER = LoggerFactory.getLogger(ComponentPageServiceImpl.class);


    @Autowired
    private RemoteComponentService remoteComponentService;

    @Autowired
    private RemotePageComponentService remotePageComponentService;

    @Resource(name = "redisTemplate")
    private AdvancedCacheClient advancedCacheClient;

    @Autowired
    private PageServiceImpl pageService;


    /**
     * 预览保存组件页面
     *
     * 校验各个组件是否有效  组件必填项是否存在
     * 将各个组件的CSS代码打包成一个文件 JS代码打包成一个文件
     * 替换pageShell.vm 的占位符 生成将新的HTML代码 获取MD5摘要 将HTML代码上传到Oss
     * 返回前端预览HTML文件的url  预览地址  MD5摘要
     * @param param
     * @return
     */
    @Override
    public ComponentPagePreviewVO preview(ComponentPagePreParam param) {

        ComponentPageProcessor pageProcessor = ComponentPageDecider.getProcessor(PageTypeEnum.findByValue(param.getPageType()));
        return pageProcessor.preview(param);
    }



    /**
     * 保存页面
     *    代码已经在预览页面时组合完成，这里只是将数据保存在数据库
     *
     * @param saveParam
     * @return
     */
    @Override
    public Boolean save(ComponentPageSaveParam saveParam) {
        Long pageId = saveParam.getId();
        //编辑的时候增加权限校验
        if(pageId != null && pageId > 0){
            pageService.checkPermission(pageId);
        }

        BaiqiPageDto pageDto = new BaiqiPageDto();
        pageDto.setId(pageId);
        pageDto.setPageImg(StringUtils.EMPTY);
        pageDto.setPageName(saveParam.getPageName());
        pageDto.setPageMd5(saveParam.getMd5());
        pageDto.setPageUrl(saveParam.getPageFileUrl());
        pageDto.setPageStatus(PageStatusEnum.VALID.getValue());
        pageDto.setPageType(saveParam.getPageType());
        if(pageId == null){
            pageDto.setCreateName(RequestTool.getAdmin().getName());
        }
        pageDto.setOperatorName(RequestTool.getAdmin().getName());
        pageDto.setStrategyTypeList(saveParam.getStrategyTypeList());
        pageDto.setPageInfo(buildJsPlatform(saveParam.getJsPlatform()));

        List<Long> componentIdList = saveParam.fetchComponentIds();
        List<ComponentDto> componentDtoList = remoteComponentService.listBatchByIds(componentIdList);
        Set<Long> componentIdSet = Sets.newHashSet(componentIdList);
        if (componentDtoList.size() != componentIdSet.size()) {
            throw new QihoManagerException("保存参数错误");
        }

        Map<Long, ComponentDto> componentMap = new HashMap<>(16);
        for (ComponentDto componentDto : componentDtoList) {
            componentMap.put(componentDto.getId(), componentDto);
        }

        List<PageComponentDto> pageComponentDtos = Lists.newArrayList();
        int count = 0;
        for (ComponentSimpleParam e :saveParam.getList()) {
            PageComponentDto pageComponentDto = new PageComponentDto();
            pageComponentDto.setConfigParam(e.getConfigParam());
            pageComponentDto.setComponentId(e.getComponentId());
            pageComponentDto.setComponentType(componentMap.get(e.getComponentId()).getComponentType());
            pageComponentDto.setSortNum(++count);
            pageComponentDtos.add(pageComponentDto);
        }

        ResultDto<Long> resultDto = remotePageComponentService.savePageComponent(pageDto, pageComponentDtos);
        if (!resultDto.isSuccess()) {
            throw new QihoManagerException(resultDto.getMsg());
        }

        // 失效页面缓存
        if (pageId != null) {
            String key = "page:id:" + pageId;
            advancedCacheClient.remove(key);
        }

        return true;
    }

    /**
     * 构建json串
     *
     * @param jsPlatform
     * @return
     */
    private String buildJsPlatform(Integer jsPlatform) {
        if(jsPlatform == null){
            return null;
        }
        // 检查类型
        JSPlatformEnum jsPlatformEnum = JSPlatformEnum.findByType(jsPlatform);
        if(jsPlatformEnum == null){
            return null;
        }
        // 序列化
        JSONObject jb = new JSONObject();
        jb.put(PageInfoJsonKeyConstant.JS_PLATFORM, jsPlatform);
        return jb.toJSONString();
    }

    /**
     * 从json串中获取js对接平台
     *
     * @param pageInfo
     * @return
     */
    private Integer getJsPlatform(String pageInfo) {
        if(StringUtils.isBlank(pageInfo)){
            return null;
        }

        JSONObject jb = JSON.parseObject(pageInfo);
        if(jb == null){
            return null;
        }

        Object v = jb.get(PageInfoJsonKeyConstant.JS_PLATFORM);
        return v != null ? Integer.valueOf(v.toString()) : null;
    }

    @Override
    public PageComponentEditVO edit(Long pageId) {
        AssertUtil.numericIsPositive(pageId, "页面id不合法");

        PageComponentDetailDto pageComponentDetail = remotePageComponentService.findPageComponentById(pageId);
        if (pageComponentDetail == null) {
            throw new QihoManagerException("页面模板不存在");
        }

        if (CollectionUtils.isEmpty(pageComponentDetail.getComponentList())) {
            throw new QihoManagerException("该页面模板没有配置组件");
        }

        PageComponentEditVO editVO = new PageComponentEditVO();
        editVO.setPageName(pageComponentDetail.getPageName());
        editVO.setJsPlatform(getJsPlatform(pageComponentDetail.getPageInfo()));
        editVO.setId(pageComponentDetail.getId());
        List<ComponentSimpleVO> componentList = pageComponentDetail.getComponentList().stream().map(e->{
            ComponentSimpleVO componentVO = BeanUtils.copy(e, ComponentSimpleVO.class);
            componentVO.setId(e.getComponentId());
            componentVO.setComponentType(e.getComponentType().getNum());
            return componentVO;
        }).collect(Collectors.toList());
        editVO.setComponentList(componentList);

        return editVO;
    }

    @Override
    public Boolean copy(ComponentPageCopyParam param) {

        AssertUtil.numericIsPositive(param.getId(), "页面id不合法");
        AssertUtil.stringNotBlank(param.getPageName(), "页面名称不能为空");

        PageComponentDetailDto pageComponentDetail = remotePageComponentService.findPageComponentById(param.getId());
        if (pageComponentDetail == null) {
            throw new QihoManagerException("页面模板不存在");
        }

        BaiqiPageDto pageDto = BeanUtils.copy(pageComponentDetail, BaiqiPageDto.class);
        pageDto.setId(null);
        pageDto.setPageName(param.getPageName());
        pageDto.setOperatorName(RequestTool.getAdmin().getName());

        // 查询页面表单必填项
        List<PageFormStrategyDto> pageFormList = remotePageComponentService.findPageFormById(param.getId());
        if (CollectionUtils.isNotEmpty(pageFormList)) {
            pageDto.setStrategyTypeList(pageFormList);
        }

        ResultDto<Long> resultDto = remotePageComponentService.savePageComponent(pageDto, pageComponentDetail.getComponentList());
        if (!resultDto.isSuccess()) {
            throw new QihoManagerException(resultDto.getMsg());
        }

        return true;
    }


}
