package com.qiho.manager.biz.service.impl.component;

import cn.com.duiba.sso.api.tool.RequestTool;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.duiba.wolf.utils.SecurityUtils;
import com.google.common.base.Charsets;
import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import com.google.common.io.*;
import com.qiho.center.api.dto.ResultDto;
import com.qiho.center.api.dto.component.ComponentDto;
import com.qiho.center.api.enums.component.ComponentTypeEnum;
import com.qiho.center.api.remoteservice.component.RemoteComponentService;
import com.qiho.manager.biz.params.component.ComponentParam;
import com.qiho.manager.biz.service.component.ComponentService;
import com.qiho.manager.biz.vo.component.ComponentVO;
import com.qiho.manager.common.constant.CacheConstantseEnum;
import com.qiho.manager.common.exception.QihoManagerException;
import com.qiho.manager.common.util.AssertUtil;
import com.qiho.manager.common.util.FileIDGenerator;
import com.qiho.manager.common.util.UploadTool;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.List;
import java.util.stream.Collectors;


/**
 * @author chensong
 * @create 2018-07-25
 */
@Service
public class ComponentServiceImpl implements ComponentService {


    private static final Logger LOGGER = LoggerFactory.getLogger(ComponentServiceImpl.class);

    /** 代码之间的分隔符 */
    private static final String CODE_SPLIT = "<split>";

    private static final String PATH_PRE = "component/";

    private static final Joiner LINE_JOINER =  Joiner.on("");

    private static final Splitter CODE_SPLITTER = Splitter.on(CODE_SPLIT);

    private static final int DEAULT_CODE_NUM = 3;

    @Value("${oss.objectPath}")
    private String ossYunUrl;

    @Resource(name = "redisTemplate")
    private AdvancedCacheClient advancedCacheClient;

    @Autowired
    private RemoteComponentService remoteComponentService;


    @Override
    public Boolean createComponent(ComponentParam param) {
        ComponentTypeEnum componentType = ComponentTypeEnum.getByNum(param.getComponentType());
        AssertUtil.objectNotNull(componentType, "组件类型不正确");

        String code = this.buildCode(param.getHtmlCode(), param.getCssCode(), param.getJsCode());
        String md5 = SecurityUtils.encode2StringByMd5(code);
        String codeUrl = uploadOss(code);
        String adminName = RequestTool.getAdmin().getName();

        ComponentDto componentDto = new ComponentDto();
        componentDto.setComponentType(componentType);
        componentDto.setComponentName(param.getComponentName());
        componentDto.setCodeUrl(codeUrl);
        componentDto.setCodeMd5(md5);
        componentDto.setConfigParam(param.getConfigParam());
        componentDto.setImage(param.getImage());
        componentDto.setOperator(adminName);

        ResultDto<Long> result = remoteComponentService.createComponent(componentDto);
        if (!result.isSuccess()) {
            throw new QihoManagerException(result.getMsg());
        }

        // 添加Redis缓存
        String key = CacheConstantseEnum.COMPONENT_FILE_URL.toString() + ":" + result.getResult();
        advancedCacheClient.set(key, code, CacheConstantseEnum.COMPONENT_FILE_URL.getTime(),
                CacheConstantseEnum.COMPONENT_FILE_URL.getTimeUnit());

        return true;
    }


    @Override
    public Boolean updateComponent(ComponentParam param) {
        Long id = param.getId();
        AssertUtil.numericIsPositive(id, "id不合法");

        ComponentDto componentDto = remoteComponentService.findById(id);
        if (componentDto == null){
            throw new QihoManagerException("组件不存在");
        }
        if (componentDto.getComponentType() != ComponentTypeEnum.getByNum(param.getComponentType())) {
            throw new QihoManagerException("组件类型不正确");
        }

        String code = buildCode(param.getHtmlCode(), param.getCssCode(), param.getJsCode());
        String newMd5 = SecurityUtils.encode2StringByMd5(code);
        if (!StringUtils.equalsIgnoreCase(newMd5, componentDto.getCodeMd5())) {
            // 代码需要重新上传到服务器
            String url = uploadOss(code);
            componentDto.setCodeUrl(url);
            componentDto.setCodeMd5(newMd5);
        }

        componentDto.setImage(param.getImage());
        componentDto.setConfigParam(param.getConfigParam());
        componentDto.setComponentName(param.getComponentName());
        componentDto.setOperator(RequestTool.getAdmin().getName());

        ResultDto<Long> result = remoteComponentService.updateComponent(componentDto);
        if (!result.isSuccess()) {
            throw new QihoManagerException(result.getMsg());
        }

        // 失效组件代码的Redis缓存
        String key = CacheConstantseEnum.COMPONENT_FILE_URL.toString() + ":" + param.getId();
        advancedCacheClient.remove(key);

        // 添加新的缓存
        advancedCacheClient.set(key, code, CacheConstantseEnum.COMPONENT_FILE_URL.getTime(),
                CacheConstantseEnum.COMPONENT_FILE_URL.getTimeUnit());

        return true;
    }


    @Override
    public ComponentVO findById(Long id) {
        AssertUtil.numericIsPositive(id, "组件id不合法");

        ComponentDto componentDto = remoteComponentService.findById(id);
        AssertUtil.objectNotNull(componentDto, "组件不存在，组件id=" + id);

        // 从缓存中获取代码
        String key = CacheConstantseEnum.COMPONENT_FILE_URL.toString() + ":" + id;
        String code = advancedCacheClient.get(key);

        ComponentVO componentVO = BeanUtils.copy(componentDto, ComponentVO.class);
        if (StringUtils.isNotBlank(code)) {
            // 从缓存中获取到代码
            List<String> codeList = CODE_SPLITTER.splitToList(code);
            if (codeList.size() != DEAULT_CODE_NUM) {
                throw new QihoManagerException("代码解析异常");
            }
            componentVO.setHtmlCode(codeList.get(0));
            componentVO.setCssCode(codeList.get(1));
            componentVO.setJsCode(codeList.get(2));

        } else {
            // 未从缓存中获取到代码 这从oss中download代码
            List<String> codeList = this.downloadCode(componentDto.getCodeUrl(), id);
            componentVO.setHtmlCode(codeList.get(0));
            componentVO.setCssCode(codeList.get(1));
            componentVO.setJsCode(codeList.get(2));
        }

        componentVO.setComponentType(componentDto.getComponentType().getNum());
        componentVO.setGmtModified(DateUtils.getSecondStr(componentDto.getGmtModified()));

        return componentVO;
    }

    @Override
    public List<ComponentVO> findByType(Integer typeNum) {
        AssertUtil.numericIsPositive(typeNum, "无效的组件类型");

        ComponentTypeEnum type = ComponentTypeEnum.getByNum(typeNum);
        AssertUtil.objectNotNull(type, "无效的组件类型 ");

        List<ComponentDto> componentDtoList = remoteComponentService.listByType(type);
        if (CollectionUtils.isEmpty(componentDtoList)) {
            return Lists.newArrayList();
        }

        List<ComponentVO> list = componentDtoList.stream().map(e -> {
            ComponentVO vo = BeanUtils.copy(e, ComponentVO.class);
            vo.setComponentType(e.getComponentType().getNum());
            vo.setGmtModified(DateUtils.getSecondStr(e.getGmtModified()));
            vo.setGmtCreate(DateUtils.getSecondStr(e.getGmtCreate()));
            return vo;
        }).collect(Collectors.toList());

        return list;
    }

    /**
     * 从Oss服务器下载代码文件  分隔成HTML css js代码
     *
     * @param codeUrl
     *      代码地址
     *
     * @return
     *      html css js 代码的集合
     */
    private List<String> downloadCode(String codeUrl, Long componentId){
        AssertUtil.stringNotBlank(codeUrl, "代码文件url为空");

        String code ;
        try {
            // 获取代码
            URL url = new URL(codeUrl);
            CharSource charset = Resources.asCharSource(url, Charsets.UTF_8);
            List<String> lines = charset.readLines();
            code = LINE_JOINER.join(lines);

            AssertUtil.stringNotBlank(code, "获取代码为空");
        } catch (IOException e) {
            LOGGER.error("代码文件获取异常, codeUrl = [{}]", codeUrl, e);
            throw new QihoManagerException("代码文件获取异常");
        }

        // 将代码拆分
        List<String> codeList = CODE_SPLITTER.splitToList(code);
        if (codeList.size() != DEAULT_CODE_NUM) {
            throw new QihoManagerException("代码解析异常");
        }

        // 代码放在Redis
        String key = CacheConstantseEnum.COMPONENT_FILE_URL.toString() + ":" + componentId;
        advancedCacheClient.set(key, code,
                CacheConstantseEnum.COMPONENT_FILE_URL.getTime(), CacheConstantseEnum.COMPONENT_FILE_URL.getTimeUnit());

        return codeList;
    }



    /**
     * 将HTML css js 代码解码后 拼接分隔符 组合一个String
     *
     * @param htmlCode
     *      Base64 加密的HTML代码
     *
     * @param cssCode
     *      Base64 加密的CSS代码
     *
     * @param jsCode
     *      Base64 加密的Js代码
     *
     * @return
     */
    private String buildCode(String htmlCode, String cssCode, String jsCode){
        StringBuilder builder = new StringBuilder();

        // 拼接HTML代码
        if (StringUtils.isNotBlank(htmlCode)) {
            builder.append(new String(SecurityUtils.decodeBase64(htmlCode)));
        }
        // 拼接分隔符
        builder.append(CODE_SPLIT);

        // 拼接css 代码
        if (StringUtils.isNotBlank(cssCode)) {
            builder.append(new String(SecurityUtils.decodeBase64(cssCode)));
        }
        // 拼接分隔符
        builder.append(CODE_SPLIT);

        // 拼接js 代码
        if (StringUtils.isNotBlank(jsCode)){
            builder.append(new String(SecurityUtils.decodeBase64(jsCode)));
        }

        return builder.toString();
    }

    /**
     * 代码写入文件 上传Oss服务器
     * 返回文件地址
     * @param code
     * @return
     */
    private String uploadOss(String code){
        if (StringUtils.isBlank(code)) {
            throw new QihoManagerException("代码为空");
        }
        String fileName = FileIDGenerator.getRandomString(10);

        try {
            File file = File.createTempFile(fileName,".txt");
            CharSink charSink = Files.asCharSink(file, Charset.defaultCharset(), FileWriteMode.APPEND);
            charSink.write(code);

            String filePath = PATH_PRE + new DateTime().toString("yyyyMMdd") + "/" +file.getName();
            String url = UploadTool.uploadOssNotCDN(file,filePath, "text/plain");

            return "http:" + url;
        } catch (Exception e) {
            LOGGER.error("上传文件错误", e);
            throw new QihoManagerException(e.getMessage());
        }
    }



}
