package com.qiho.manager.biz.service.item;

import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Objects;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.remoteservice.item.RemoteItemEvaluateBackendService;
import com.qiho.manager.biz.runnable.AbstractItemEvaluateHandler;
import com.qiho.manager.biz.vo.item.ItemEvaluateDealWithVO;
import com.qiho.manager.common.constant.CacheConstantseEnum;
import com.qiho.manager.common.enums.ItemEvaluateSaveFailedEnum;
import com.qiho.manager.common.util.OSSFileService;
import com.qiho.manager.common.util.ReadExcelUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.InputStream;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @author jiafeng.geng
 * @date 2018-8-30
 */
@Service
public class ItemEvaluateDealWithService implements ApplicationContextAware {

    private Logger logger = LoggerFactory.getLogger(ItemEvaluateDealWithService.class);

    private static final int MAX_SHOW_LEN = 50;

    private CacheConstantseEnum constantse = CacheConstantseEnum.IMPORT_ITEM_EVALUATES;

    private ApplicationContext applicationContext;

    @Autowired
    private OSSFileService ossFileService;

    @Autowired
    private RemoteItemEvaluateBackendService remoteItemEvaluateBackendService;

    @Resource(name = "stringRedisTemplate")
    private StringRedisTemplate redisTemplate;

    /**
     * 提交
     *
     * @param fileUrl      文件url
     * @param itemId       商品id
     * @param handlerClass
     */
    public ItemEvaluateDealWithVO submitDealWithTask(String fileUrl, Long itemId,
                                                     Class<? extends AbstractItemEvaluateHandler> handlerClass) {
        ItemEvaluateDealWithVO vo = new ItemEvaluateDealWithVO();
        InputStream input = ossFileService.getOssFileInputStream(fileUrl);
        //加载文件
        try {
            if (Objects.equal(null, input)) {
                throw new QihoException("文件不存在");
            }

            String taskId = createTaskId();
            String cacheKey = constantse.getCacheKey(taskId);
            List<String> evalList = ReadExcelUtil.createExcel(input, ReadExcelUtil.isExcel2003(fileUrl));
            AbstractItemEvaluateHandler handler = applicationContext.getBean(handlerClass);

            int count = flush(cacheKey, evalList, handler, itemId);
            if (count == ItemEvaluateSaveFailedEnum.EXCESS_MAX_THRESHOLD.getCode()) {
                vo.setCount(ItemEvaluateSaveFailedEnum.EXCESS_MAX_THRESHOLD.getCode());
                return vo;
            }

            redisTemplate.boundHashOps(cacheKey).put("count", String.valueOf(count));

            if (count == 0) {
                redisTemplate.boundHashOps(cacheKey).put("success", "true");
            }

            // 3小时后删除在缓存中的任务元数据
            redisTemplate.expire(cacheKey, 3, TimeUnit.HOURS);

            vo.setTaskId(taskId);
            vo.setCount(count);
        } catch (Exception e) {
            logger.error("导入任务失败", e);
        } finally {
            ossFileService.closeInputStream(input);
        }
        return vo;
    }

    private <T> Integer flush(String cacheKey, List<String> lines, AbstractItemEvaluateHandler<T> handler,
                              Long itemId) {
        if (lines.isEmpty()) {
            return 0;
        }

        // 当前商品的有效评价总数
        Integer evalCount = remoteItemEvaluateBackendService.queryItemEvalCount(itemId);
        if (evalCount >= MAX_SHOW_LEN) {
            return ItemEvaluateSaveFailedEnum.EXCESS_MAX_THRESHOLD.getCode();
        }

        List<T> list = Lists.newArrayList();
        for (String line : lines) {
            T params = handler.transform(line, itemId);
            if (params != null) {
                list.add(params);
            }
            if (evalCount + list.size() >= MAX_SHOW_LEN) {
                break;
            }
        }
        int total = 0;
        if (list.isEmpty()) {
            return total;
        }
        total = handler.exeTask(cacheKey, list);
        lines.clear();
        return total;
    }

    /**
     * 生成任务Id
     *
     * @return
     */
    private String createTaskId() {
        Long taskId = System.currentTimeMillis();
        String key = constantse.getCacheKey(taskId);
        Map<String, String> info = Maps.newHashMap();
        info.put("success", "false");
        info.put("count", "0");
        info.put("successCount", "0");
        info.put("failCount", "0");
        info.put("taskCount", "0");
        // 失败的数据id,逗号分隔,用于标识处理数据时哪一条数据处理失败了
        info.put("failIds", "");
        redisTemplate.opsForHash().putAll(key, info);
        return taskId.toString();
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
    }

    public JSONObject findImportWithInfo(String taskId) {
        String cacheKey = constantse.getCacheKey(taskId);
        BoundHashOperations<String, String, Object> operations = redisTemplate.boundHashOps(cacheKey);

        JSONObject json = new JSONObject();
        Map<String, Object> info = operations.entries();
        if (info.isEmpty()) {
            json.put("success", true);
        } else {
            json.putAll(info);
            if (json.getInteger("count") != 0 && json.getInteger("taskCount") == 0) {
                operations.put("success", "true");
                redisTemplate.delete(taskId);
            }
        }

        json.put("success", json.getBoolean("success"));
        return json;
    }

}
