package com.qiho.manager.biz.service.open.impl;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.wolf.dubbo.DubboResult;
import com.qiho.center.api.dto.OpenInterfaceDto;
import com.qiho.center.api.dto.OrderSnapshotDto;
import com.qiho.center.api.dto.agent.BaiqiAgentDto;
import com.qiho.center.api.dto.logistics.LogisticsDto;
import com.qiho.center.api.dto.order.OrderCancelDto;
import com.qiho.center.api.dto.order.OrderConfirmDto;
import com.qiho.center.api.enums.log.OrderLogSubTypeEnum;
import com.qiho.center.api.params.OrderSubTypeParam;
import com.qiho.center.api.remoteservice.agent.RemoteBaiqiAgentBackendService;
import com.qiho.center.api.remoteservice.agent.RemoteBaiqiAgentService;
import com.qiho.center.api.remoteservice.logistics.RemoteLogisticsService;
import com.qiho.center.api.remoteservice.order.RemoteOrderService;
import com.qiho.manager.biz.params.open.OpenOrderUpdateStatusParam;
import com.qiho.manager.biz.service.open.OpenOrderService;
import com.qiho.manager.common.enums.OpenOrderStatusEnum;
import com.qiho.manager.common.exception.QihoManagerException;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.Objects;

@Service
public class OpenOrderServiceImpl implements OpenOrderService {

    private Logger LOGGER = LoggerFactory.getLogger(OpenOrderServiceImpl.class);

    @Resource
    private RemoteBaiqiAgentService agentService;

    @Resource
    private RemoteBaiqiAgentBackendService remoteBaiqiAgentBackendService;

    @Resource
    private RemoteOrderService remoteOrderService;

    @Resource
    private RemoteLogisticsService remoteLogisticsService;

    @Override
    public void updateOrderStatus(OpenOrderUpdateStatusParam statusParam) {

        // 1、参数校验
        updateOrderStatusParamCheck(statusParam);

        // 2、审核成功
        if(Objects.equals(OpenOrderStatusEnum.AUDIT_PASS.getStatus(), statusParam.getStatus())){
            OrderSubTypeParam subTypeParam = new OrderSubTypeParam();
            subTypeParam.setOrderId(statusParam.getOrderId());
            subTypeParam.setSubType(OrderLogSubTypeEnum.AUDIT_PASS.getSubType());
            remoteOrderService.recordLog(subTypeParam);
            return;
        }

        // 3、审核失败
        if(Objects.equals(OpenOrderStatusEnum.AUDIT_FAIL.getStatus(), statusParam.getStatus())){
            OrderSubTypeParam subTypeParam = new OrderSubTypeParam();
            subTypeParam.setOrderId(statusParam.getOrderId());
            subTypeParam.setSubType(OrderLogSubTypeEnum.AUDIT_FAIL.getSubType());
            remoteOrderService.recordLog(subTypeParam);
            return;
        }

        // 4、发货
        if(Objects.equals(OpenOrderStatusEnum.SEND_OUT.getStatus(), statusParam.getStatus())){
            confirmOrder(statusParam);
            return;
        }

        // 4、取消发货
        if(Objects.equals(OpenOrderStatusEnum.CANCEL_SEND.getStatus(), statusParam.getStatus())){
            cancelOrder(statusParam);
            return;
        }

        // 5、更新物流
        if(Objects.equals(OpenOrderStatusEnum.UPDATE_LOGISTICS.getStatus(), statusParam.getStatus())){
            orderUpdate(statusParam);
            return;
        }

        throw new QihoManagerException("未知的订单状态");
    }
    private void confirmOrder(OpenOrderUpdateStatusParam statusParam){
        OrderConfirmDto dto = new OrderConfirmDto();
        dto.setOrderId(statusParam.getOrderId());
        dto.setExpress(statusParam.getLogisticsCode());
        dto.setExpressCode(statusParam.getExpressCode());
        try {
            Boolean result = remoteOrderService.orderConfirm(dto);
            if(!result){
                throw new QihoManagerException("remoteOrderService.orderConfirm fail");
            }
        } catch (BizException e) {
            throw new QihoManagerException(e.getMessage());
        }
    }

    private void cancelOrder(OpenOrderUpdateStatusParam statusParam){
        OrderCancelDto dto = new OrderCancelDto();
        dto.setOrderId(statusParam.getOrderId());
        dto.setRemark(statusParam.getRemark());
        try {
            Boolean result = remoteOrderService.orderCancel(dto);
            if(!result){
                throw new QihoManagerException("remoteOrderService.orderCancel fail");
            }
        } catch (BizException e) {
            throw new QihoManagerException(e.getMessage());
        }
    }

    private void orderUpdate(OpenOrderUpdateStatusParam statusParam){
        OrderConfirmDto dto = new OrderConfirmDto();
        dto.setOrderId(statusParam.getOrderId());
        dto.setExpress(statusParam.getLogisticsCode());
        dto.setExpressCode(statusParam.getExpressCode());
        try {
            Boolean result = remoteOrderService.orderUpdate(dto);
            if(!result){
                throw new QihoManagerException("remoteOrderService.orderUpdate fail");
            }
        } catch (BizException e) {
            throw new QihoManagerException(e.getMessage());
        }
    }

    private void updateOrderStatusParamCheck(OpenOrderUpdateStatusParam statusParam){
        // 1、请求参数校验
        Assert.notNull(statusParam, "请求参数不能为空");
        Assert.notNull(statusParam.getAppKey(), "appKey不能为空");
        Assert.notNull(statusParam.getAppSecret(), "appSecret不能为空");
        Assert.notNull(statusParam.getMerchantId(), "merchantId不能为空");
        Assert.notNull(statusParam.getOrderId(), "orderId不能为空");
        Assert.notNull(statusParam.getStatus(), "status不能为空");

        // 2、商户校验
        BaiqiAgentDto agentDto = remoteBaiqiAgentBackendService.findAgentByMerchantId(statusParam.getMerchantId());
        Assert.notNull(agentDto, "没有操作权限");
        OpenInterfaceDto interfaceDto =  agentService.findByAgentId(agentDto.getId());
        Assert.notNull(interfaceDto, "没有操作权限");
        Assert.isTrue(interfaceDto.getAppkey().equals(statusParam.getAppKey()) && interfaceDto.getAppsecret().equals(statusParam.getAppSecret()),
                "没有操作权限");

        // 3、物流校验
        if(StringUtils.isNotBlank(statusParam.getLogisticsCode())){
            LogisticsDto dto = remoteLogisticsService.findByCode(statusParam.getLogisticsCode());
            if(Objects.isNull(dto) || (!Objects.equals(dto.getLogisticsName(), statusParam.getLogisticsName()))){
                throw new QihoManagerException("物流公司信息不正确");
            }
        }


        // 4、订单校验
        DubboResult<OrderSnapshotDto> result = remoteOrderService.findOrderSnapshot(statusParam.getOrderId());
        if (!result.isSuccess()) {
            LOGGER.error("findByOrderId error, orderId = {}, errorCode = {}, errorMsg = {}", statusParam.getOrderId(), result.getReturnCode(), result.getMsg());
            throw new QihoManagerException("系统异常");
        }
        if(Objects.isNull(result.getResult()) || (!statusParam.getMerchantId().equals(String.valueOf(result.getResult().getMerchantId())))){
            throw new QihoManagerException("该订单不属于当前商家");
        }
    }
}
