package com.qiho.manager.biz.service.page.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

import javax.annotation.Resource;

import cn.com.duiba.sso.api.domain.dto.AdminDto;
import cn.com.duiba.sso.api.tool.RequestTool;
import cn.com.duiba.sso.api.web.power.AdminPowerCacheService;
import com.qiho.center.api.dto.ResultDto;
import com.qiho.center.api.dto.page.PageComponentDetailDto;
import com.qiho.center.api.dto.page.PageFormStrategyDto;
import com.qiho.center.api.enums.page.PageTypeReflectEnum;
import com.qiho.center.api.params.param.PagePagingParam;
import com.qiho.center.api.remoteservice.page.RemotePageComponentService;
import com.qiho.manager.biz.params.page.LandPageSaveParam;
import com.qiho.manager.common.constant.DomainConstantUtil;
import com.qiho.manager.common.enums.DataPermissionEnum;
import com.qiho.manager.common.enums.ErrorCode;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import com.google.common.base.Charsets;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.google.common.io.CharSink;
import com.google.common.io.CharSource;
import com.google.common.io.FileWriteMode;
import com.google.common.io.Files;
import com.google.common.io.Resources;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.page.BaiqiPageDto;
import com.qiho.center.api.enums.page.PageStatusEnum;
import com.qiho.center.api.enums.page.PageTypeEnum;
import com.qiho.center.api.remoteservice.page.RemotePageBackendService;
import com.qiho.manager.biz.params.page.PageSaveParam;
import com.qiho.manager.biz.service.page.PageService;
import com.qiho.manager.biz.vo.Pagenation;
import com.qiho.manager.biz.vo.page.PageEditVO;
import com.qiho.manager.biz.vo.page.PagePagingVO;
import com.qiho.manager.common.exception.QihoManagerException;
import com.qiho.manager.common.util.UploadTool;

import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.duiba.wolf.utils.SecurityUtils;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author peanut.huang
 * @date 2018/2/24.
 */
@Service("pageService")
public class PageServiceImpl implements PageService {

    private static final Logger logger = LoggerFactory.getLogger(PageServiceImpl.class);

    @Resource
    private RemotePageBackendService      remotePageBackendService;
    @Resource(name = "redisTemplate")
    private AdvancedCacheClient           advancedCacheClient;
    @Value("${oss.objectPath}")
    private String                        ossYunUrl;

    @Autowired
    private AdminPowerCacheService adminPowerCacheService;

    @Autowired
    private RemotePageComponentService remotePageComponentService;

    @Override
    public Pagenation<PagePagingVO> find4Paging(String pageName, List<Integer> pageTypes,String tabName, Integer offset, Integer pageSize) {
        PagePagingParam pagePagingParam = buildQueryParam(pageName,pageTypes,tabName ,offset,pageSize);

        PagenationDto<BaiqiPageDto> pagenationDto;
        try {
            pagenationDto = remotePageBackendService.find4Paging(pagePagingParam);
        }catch (Exception e){
            throw new QihoManagerException("页面分页查询错误", e);
        }
        Pagenation<PagePagingVO> result = new Pagenation<>();

        //count
        int total = pagenationDto.getTotal();

        //list
        List<BaiqiPageDto> pageDtoList = pagenationDto.getList();

        result.setTotal(total);
        result.setList(transDto2Vo(pageDtoList));

        return result;
    }

    private PagePagingParam buildQueryParam(String pageName, List<Integer> pageTypes, String tabName, Integer offset, Integer pageSize) {

        PagePagingParam pagePagingParam = new PagePagingParam();
        pagePagingParam.setPageName(pageName);
        pagePagingParam.setPageTypes(pageTypes);
        pagePagingParam.setPageSize(pageSize);
        pagePagingParam.setOffset(offset);
        boolean myLandPage = Objects.equals(tabName, PageTypeReflectEnum.MY_LAND_PAGE.getCode());
        if(myLandPage){
            pagePagingParam.setOriginType(PageTypeEnum.MY_LAND_PAGE.getValue());
            pagePagingParam.setPageTypes(null);
        }else {
            pagePagingParam.setOriginType(0);
        }
        String permissionCode = getPermissionCodeByTypes(pageTypes.get(0), false);

        //设置创建者的姓名
        buildCreateName(pagePagingParam, permissionCode);

        return pagePagingParam;
    }


    /**
     * 根据用户是否有全局权限 来设置创建者的姓名
     *
     * @param pagePagingParam
     */
    private void buildCreateName(PagePagingParam pagePagingParam, String permissionCode) {

        boolean hasThePermission = hasThePermission(permissionCode);

        //如果用户没有全局权限 那么查询的时候只能查询自己创建的落地页
        if (!hasThePermission) {
            pagePagingParam.setCreateName(RequestTool.getAdmin().getName());
        }
    }



    /**
     * dto 2 vo
     *
     * @param pageDtoList
     * @return
     */
    private List<PagePagingVO> transDto2Vo(List<BaiqiPageDto> pageDtoList) {
        if(CollectionUtils.isEmpty(pageDtoList)){
            return Collections.emptyList();
        }

        List<PagePagingVO> result = Lists.newArrayListWithExpectedSize(pageDtoList.size());
        pageDtoList.forEach( e -> {

            PagePagingVO pagingVO = BeanUtils.copy(e, PagePagingVO.class);
            pagingVO.setGmtCreate(DateUtils.getSecondStr(e.getGmtCreate()));
            pagingVO.setGmtModified(DateUtils.getSecondStr(e.getGmtModified()));

            result.add(pagingVO);

        });

        return result;
    }

    @Override
    public void savePage(String operatorName, PageSaveParam pageSaveParam) {

        //编辑的时候新增权限校验
        Long pageId = pageSaveParam.getId();
        if(pageId  != null && pageId > 0){
            checkPermission(pageId);
        }
        //dto
        BaiqiPageDto pageDto = new BaiqiPageDto();
        if(pageSaveParam.getId() == null){
            //创建者姓名
            pageDto.setCreateName(operatorName);
        }
        pageDto.setOperatorName(operatorName);
        pageDto.setId(pageSaveParam.getId());
        pageDto.setPageType(pageSaveParam.getPageType());
        pageDto.setPageName(pageSaveParam.getPageName());
        pageDto.setPageImg(pageSaveParam.getPageImg());

        //代码生成文件
        File tempFile = createFileFromCode(pageDto, pageSaveParam.getCode());

        //文件上传云，得到云上路径
        upload2Oss(pageDto, tempFile, pageSaveParam.getPageType());

        //文件保存db
        doSave(pageDto);

        // 失效redis 页面代码
        expireRedis(pageDto.getId());
    }

    /**
     * 权限校验
     *
     * @param pageId  页面Id
     */
    public void checkPermission(Long pageId) {
        BaiqiPageDto baiqiPageDto = remotePageBackendService.findById(pageId);
        if (baiqiPageDto == null) {
            throw new QihoManagerException("更新的页面不存在");
        }
        checkSsoPermission(pageId, getPermissionCodeByTypes(baiqiPageDto.getPageType(), true));
    }


    private void checkSsoPermission(Long pageId,String permissionCode){
        boolean hasThePermission = hasThePermission(permissionCode);
        if(!hasThePermission){
            BaiqiPageDto baiqiPageDto = remotePageBackendService.findById(pageId);
            if(baiqiPageDto == null || !baiqiPageDto.getCreateName().equals(RequestTool.getAdmin().getName())){
                throw new QihoManagerException(ErrorCode.NOT_ALLOWED.getMsg());
            }
        }
    }

    private void expireRedis(Long pageId) {
        String key = "page:id:" + pageId;
        advancedCacheClient.remove(key);
    }

    @Override
    public PageEditVO findById(Long pageId) {
        if(pageId == null || pageId <= 0){
            return null;
        }

        BaiqiPageDto pageDto ;

        try {
            pageDto = remotePageBackendService.findById(pageId);
        }catch (Exception e){
            logger.error("id[{}]查询页面出错", e);
            return null;
        }

        PageEditVO pageEditVO = BeanUtils.copy(pageDto, PageEditVO.class);

        //从地址获取代码
        String code = fetchCodeFromUrl(pageDto.getPageUrl());

        pageEditVO.setCode(code);

        return pageEditVO;
    }

    /**
     * 从地址获取代码
     *
     * @param pageUrl 页面地址
     * @return        页面代码
     */
    private String fetchCodeFromUrl(String pageUrl) {

        //get from redis
        String codes = advancedCacheClient.get(pageUrl);

        if(StringUtils.isBlank(codes)){

            try {
                URL url = new URL(pageUrl);
                CharSource charset = Resources.asCharSource(url, Charsets.UTF_8);

                //read lines
                List<String> lines = charset.readLines();

                //to string
                codes = Joiner.on("").join(lines);

                //set redis
                advancedCacheClient.set(pageUrl, codes, 5, TimeUnit.MINUTES);

                return codes;
            } catch (IOException e) {
                logger.error("从地址[{}]解析代码出错", pageUrl, e);
            }
        }

        return codes;

    }

    /**
     * 保存
     *
     * @param pageDto
     */
    private void doSave(BaiqiPageDto pageDto) {

        PageTypeEnum typeEnum = PageTypeEnum.findByValue(pageDto.getPageType());

        switch (typeEnum){

            //商品详情默认生效
            case ITEM_DETAIL:
                pageDto.setPageStatus(PageStatusEnum.VALID.getValue());
                break;

            default:
                break;
        }

        try{
           Long pageId = remotePageBackendService.savePage(pageDto);
           pageDto.setId(pageId);
        }catch (Exception e){
            logger.error("保存页面出错", e);
        }

    }

    /**
     * 文件上传至oss
     *
     *
     * @param pageDto
     * @param tempFile  临时文件
     * @param pageType  页面类型
     * @return
     */
    private void upload2Oss(BaiqiPageDto pageDto, File tempFile, int pageType) {
        if(tempFile == null){
            return;
        }

        String path = "pages";

        PageTypeEnum typeEnum = PageTypeEnum.findByValue(pageType);
        if(typeEnum != null){
            path = path + "/" + typeEnum.toString();
        }

        //path
        path = path + "/" + tempFile.getName();

        //upload
        Boolean result = UploadTool.uploadOssHtml(tempFile, path);

        if(!result){
            throw new QihoManagerException("文件上传失败");
        }

        pageDto.setPageUrl("http:" + ossYunUrl + path);
    }

    /**
     * 代码放入文件中
     *
     * <p>对比文件md5来判断代码是否变更</p>
     *
     *
     * @param pageDto
     * @param code    代码
     * @return        返回文件或者代码未变更，异常时返回null
     */
    private File createFileFromCode(BaiqiPageDto pageDto, String code) {

        Long pageId = pageDto.getId();

        //随机生成文件名
        String fileName = RandomStringUtils.randomAlphabetic(6);

        //解码后的代码
        String decodeCode = new String(SecurityUtils.decodeBase64(code));

        try {
            File file = File.createTempFile(fileName,".html");
            CharSink charSink = Files.asCharSink(file, Charset.defaultCharset(), FileWriteMode.APPEND);
            charSink.write(decodeCode);

            //本次页面代码md5
            String md5 = DigestUtils.md5Hex(new FileInputStream(file));
            pageDto.setPageMd5(md5);

            if(pageId != null){

                //源页面的md5
                String pageMd5 = remotePageBackendService.findPageMd5(pageId);

                //md5值相同说明代码未变动
                if(StringUtils.equals(pageMd5, md5)){
                    return null;
                }
            }

            return file;

        } catch (IOException e) {
            logger.error("", e);
            return null;
        }
    }

    @Override
    public Pagenation<PagePagingVO> findByPageId(Long pageId, List<Integer> pageTypes) {
        if(pageId == null || pageId <= 0){
            return null;
        }

        BaiqiPageDto pageDto ;

        try {
            pageDto = remotePageBackendService.findById(pageId);
        } catch (Exception e) {
            logger.error("pageId[{}]查询页面出错", pageId, e);
            return null;
        }

        if (null == pageDto) {
            return null;
        }

        // 判断查询到的对象是否为当前请求的页面类型
        if (!pageTypes.contains(pageDto.getPageType())) {
            return null;
        }

        boolean owner  = RequestTool.getAdmin().getName().equals(pageDto.getCreateName());

        //如果当前页面不是当前登陆创建 且不拥有全局权限 则不展示
        if (!owner && !hasThePermission(getPermissionCodeByTypes(pageTypes.get(0), true))) {
            return null;
        }
        // 转换对象类型并存储在list集合中
        List<PagePagingVO> pagePagingVOList = new ArrayList<>();
        pagePagingVOList.add(convertVO(pageDto));

        // 封装返回对象
        Pagenation<PagePagingVO> pagePagingVOPagenation = new Pagenation<>();
        pagePagingVOPagenation.setTotal(1);
        pagePagingVOPagenation.setList(pagePagingVOList);

        return pagePagingVOPagenation;
    }

    @Override
    @Transactional
    public boolean saveLandPage(LandPageSaveParam landPageSaveParam) {

        //首先保存落地页模板相关的数据
        Long pageId = landPageSaveParam.getPageId();
        Long itemId = landPageSaveParam.getItemId();
        BaiqiPageDto baiqiPageDto = remotePageBackendService.findById(pageId);

        Integer pageType;
        if(baiqiPageDto == null || (pageType = baiqiPageDto.getPageType()) == null){
            throw new QihoManagerException("页面Id错误");
        }

        Long newPageId;
        if(pageType == PageTypeEnum.COMPONENT_ITEM_DETAIL.getValue() || pageType == PageTypeEnum.COMPONENT_COLLECTION_PAGE.getValue()){
            //复制组件
            newPageId = copyComponent(landPageSaveParam.getPageId() ,baiqiPageDto.getPageName());
        }else {
            //复制单个页面模版
            newPageId = copyPage(baiqiPageDto);
        }

        BaiqiPageDto param = new BaiqiPageDto();
        //设置 落地页链接
        param.setId(newPageId);
        param.setPageType(pageType);
        param.setPageName(baiqiPageDto.getPageName());
        param.setLandPageUrl(renderGoods(newPageId,itemId));
        //更新上面保存的落地页
        remotePageBackendService.savePage(param);

        return true;
    }

    private Long copyPage(BaiqiPageDto baiqiPageDto) {
        //设置落地页type
        baiqiPageDto.setId(null);
        baiqiPageDto.setOriginType(PageTypeEnum.MY_LAND_PAGE.getValue());
        baiqiPageDto.setPageType(baiqiPageDto.getPageType());
        baiqiPageDto.setOperatorName(RequestTool.getAdmin().getName());
        baiqiPageDto.setCreateName(RequestTool.getAdmin().getName());
        return remotePageBackendService.savePage(baiqiPageDto);
    }

    private Long copyComponent(Long pageId, String pageName) {
        PageComponentDetailDto pageComponentDetail = remotePageComponentService.findPageComponentById(pageId);
        if (pageComponentDetail == null) {
            throw new QihoManagerException("页面模板不存在");
        }

        BaiqiPageDto pageDto = BeanUtils.copy(pageComponentDetail, BaiqiPageDto.class);
        pageDto.setId(null);
        pageDto.setPageName(pageName);
        pageDto.setOperatorName(RequestTool.getAdmin().getName());
        pageDto.setOriginType(PageTypeEnum.MY_LAND_PAGE.getValue());
        pageDto.setPageType(pageDto.getPageType());
        pageDto.setOperatorName(RequestTool.getAdmin().getName());
        pageDto.setCreateName(RequestTool.getAdmin().getName());

        // 查询页面表单必填项
        List<PageFormStrategyDto> pageFormList = remotePageComponentService.findPageFormById(pageId);
        if (CollectionUtils.isNotEmpty(pageFormList)) {
            pageDto.setStrategyTypeList(pageFormList);
        }

        ResultDto<Long> resultDto = remotePageComponentService.savePageComponent(pageDto, pageComponentDetail.getComponentList());
        if (!resultDto.isSuccess()) {
            throw new QihoManagerException(resultDto.getMsg());
        }
        return resultDto.getResult();
    }

    private String renderGoods(Long newPageId, Long itemId) {
        if(newPageId == null || newPageId <=0){
            throw new QihoManagerException("页面id错误");
        }

        if(itemId == null || itemId <= 0){
            throw new QihoManagerException("商品id错误");
        }

        //web host
        String host = DomainConstantUtil.getQihoWebUrl();

        return host +
                "/item/detail2?id=" +
                itemId +
                "&pid=" +
                newPageId;
    }

    private PagePagingVO convertVO(BaiqiPageDto pageDto) {
        PagePagingVO pagePagingVO = BeanUtils.copy(pageDto, PagePagingVO.class);
        pagePagingVO.setGmtCreate(DateUtils.getSecondStr(pageDto.getGmtCreate()));
        pagePagingVO.setGmtModified(DateUtils.getSecondStr(pageDto.getGmtModified()));
        return pagePagingVO;
    }

    private boolean hasThePermission(String code){
        AdminDto admin = RequestTool.getAdmin();
        if(admin == null){
            throw new QihoManagerException(ErrorCode.BAD_REQUEST.getMsg());
        }
        return adminPowerCacheService.hasPower(admin.getId(), code);
    }

    /**
     *
     * 根据页面类型 获取权限码
     *
     * @param pageType 页面类型
     * @param edit 查看还是编辑
     * @return
     */
    private String getPermissionCodeByTypes(Integer pageType,boolean edit){

        if(pageType == null){
            return StringUtils.EMPTY;
        }
        //落地页全局权限
        if(pageType == PageTypeEnum.MY_LAND_PAGE.getValue()){
            return DataPermissionEnum.MY_LAND_PAGE_ALL.getCode();
        }

        boolean isCollection = pageType == PageTypeEnum.COMPONENT_COLLECTION_PAGE.getValue() ||
                                pageType == PageTypeEnum.ITEM_COLLECTION.getValue();
        //我的集合页查看权限
        if(isCollection){
            return edit ? DataPermissionEnum.COLLECTION_EDIT_ALL.getCode() : DataPermissionEnum.COLLECTION_SEARCH_ALL.getCode();
        }

        //落地页模版权限
//        if(pageType == PageTypeEnum.ITEM_DETAIL.getValue() || pageType == PageTypeEnum.COMPONENT_ITEM_DETAIL.getValue()){
//            return DataPermissionEnum.LAND_PAGE_MODEL_ALL.getCode();
//        }
        return StringUtils.EMPTY;
    }
}
