package com.qiho.manager.common.util;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFDataFormatter;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.MultipartFile;

import com.google.common.collect.Lists;

/**
 * @author jiafeng.geng
 * @date 2018-9-3
 */
public class ReadExcelUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(ReadExcelUtil.class);

    private static boolean IS_EXCEL_2003 = true;

    private static boolean IS_EXCEL_2007 = false;

    private ReadExcelUtil() {
    }

    /**
     * 读EXCEL文件，获取信息集合
     *
     * @param mFile
     * @return
     */
    public static List<String> getExcelInfo(MultipartFile mFile) {
        // 获取文件名
        String fileName = mFile.getOriginalFilename();
        try {
            // 验证文件名是否合格
            if (!validateExcel(fileName)) {
                return Lists.newArrayList();
            }
            // 根据文件名判断文件是2003版本还是2007版本
            if (isExcel2007(fileName)) {
                IS_EXCEL_2003 = false;
                IS_EXCEL_2007 = true;
            }
            return createExcel(mFile.getInputStream(), IS_EXCEL_2003);
        } catch (Exception e) {
            LOGGER.info("读EXCEL文件，获取信息集合");
        }
        return Lists.newArrayList();
    }

    /**
     * 根据excel里面的内容读取信息
     *
     * @param is          输入流
     * @param isExcel2003 excel是2003还是2007版本
     * @return
     * @throws IOException
     */
    public static List<String> createExcel(InputStream is, boolean isExcel2003) {
        try {
            Workbook wb;
            // 当excel是2003时,创建excel2003
            if (isExcel2003) {
                wb = new HSSFWorkbook(is);
            } else {// 当excel是2007时,创建excel2007
                IS_EXCEL_2003 = false;
                IS_EXCEL_2007 = true;
                wb = new XSSFWorkbook(is);
            }
            // 读取Excel里面的信息
            return readExcelValue(wb);
        } catch (IOException e) {
            LOGGER.info("根据excel里面内容读取信息异常");
        }
        return Lists.newArrayList();
    }

    /**
     * 读取Excel里面的信息
     *
     * @param wb
     * @return
     */
    private static List<String> readExcelValue(Workbook wb) {
        // 得到第一个shell
        Sheet sheet = wb.getSheetAt(0);
        // 得到Excel的行数
        int totalRows = sheet.getPhysicalNumberOfRows();
        List<String> list = new ArrayList<>();
        // 循环Excel行数
        for (int r = 1; r <= totalRows; r++) {
            Row row = sheet.getRow(r);
            if (row != null && !hasNullCell(row)) {
                // 添加到list
                list.add(getRowStr(row, row.getPhysicalNumberOfCells()));
            }
        }
        return list;
    }

    /**
     * 判断行内是否有空的格
     *
     * @param row
     * @return
     */
    private static boolean hasNullCell(Row row) {
        int num = 0;
        Iterator<Cell> iterator = row.iterator();

        // 2003版本
        if (IS_EXCEL_2003) {
            while (iterator.hasNext()) {
                Cell c = iterator.next();
                if (c.getCellType() != HSSFCell.CELL_TYPE_BLANK) {
                    num++;
                }
            }
        }

        // 2007版本
        if (IS_EXCEL_2007) {
            while (iterator.hasNext()) {
                Cell c = iterator.next();
                if (c.getCellType() != XSSFCell.CELL_TYPE_BLANK) {
                    num++;
                }
            }
        }

        return num != row.getLastCellNum();
    }

    private static String getRowStr(Row row, int totalCells) {
        StringBuilder stringBuilder = new StringBuilder();
        for (int i = 0; i < totalCells; i++) {
            String cellStr = convertToStr(row.getCell(i));
            stringBuilder.append(cellStr).append(",");
            if (i == totalCells - 1) {
                stringBuilder.delete(stringBuilder.length() - 1, stringBuilder.length());
            }
        }
        return stringBuilder.toString();
    }

    /**
     * 将cell转换成字符串
     *
     * @param cell
     * @return
     */
    private static String convertToStr(Cell cell) {
        String cellStr = cell.toString();

        // 2003版本
        if (IS_EXCEL_2003) {
            switch (cell.getCellType()) {
            case HSSFCell.CELL_TYPE_STRING:
                cellStr = cell.getStringCellValue();
                break;
            case HSSFCell.CELL_TYPE_FORMULA:
                cellStr = cell.getCellFormula();
                break;
            case HSSFCell.CELL_TYPE_NUMERIC:
                HSSFDataFormatter dataFormatter = new HSSFDataFormatter();
                cellStr = dataFormatter.formatCellValue(cell);
                break;
            case HSSFCell.CELL_TYPE_ERROR:
                break;
            default:
            }
        }

        // 2007版本
        if (IS_EXCEL_2007) {
            switch (cell.getCellType()) {
            case XSSFCell.CELL_TYPE_STRING:
                cellStr = cell.getStringCellValue();
                break;
            case XSSFCell.CELL_TYPE_FORMULA:
                cellStr = cell.getCellFormula();
                break;
            case XSSFCell.CELL_TYPE_NUMERIC:
                HSSFDataFormatter dataFormatter = new HSSFDataFormatter();
                cellStr = dataFormatter.formatCellValue(cell);
                break;
            case XSSFCell.CELL_TYPE_ERROR:
                break;
            default:
            }
        }

        return cellStr;
    }

    /**
     * 验证EXCEL文件
     *
     * @param filePath
     * @return
     */
    private static boolean validateExcel(String filePath) {
        boolean validateResult = StringUtil.isEmpty(filePath) || !(isExcel2003(filePath) || isExcel2007(filePath));
        if (validateResult) {
            LOGGER.info("文件名不是excel格式, filePath={}", filePath);
            return false;
        }
        return true;
    }

    /**
     * 是否是2003的excel，返回true是2003
     */
    public static boolean isExcel2003(String filePath) {
        return "^.+\\.(?i)(xls)$".matches(filePath);
    }

    /**
     * 是否是2007的excel，返回true是2007
     */
    public static boolean isExcel2007(String filePath) {
        return "^.+\\.(?i)(xlsx)$".matches(filePath);
    }

}