package com.qiho.manager.common.util;

import com.aliyun.api.AliyunClient;
import com.aliyun.api.cdn.cdn20141111.request.RefreshObjectCachesRequest;
import com.aliyun.api.cdn.cdn20141111.response.RefreshObjectCachesResponse;
import com.aliyun.oss.ClientException;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.OSSException;
import com.aliyun.oss.model.ObjectMetadata;
import com.qiho.manager.common.exception.QihoManagerException;
import com.taobao.api.ApiException;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * ClassName: UploadTool <br/>
 * date: 2016年12月13日 上午11:38:32 <br/>
 * 阿里云 对象存储服务
 * @version
 */
public class UploadTool {

    private static final Logger LOG = LoggerFactory.getLogger(UploadTool.class);

    private static OSSClient    ossClient;
    private static AliyunClient cdnClient;
    private static String       bucketName;
    private static String       objectPath;                                     // 外网地址前缀

    /**
     * 上传Oss文件
     * 
     * @param file
     * @param objectName
     * @return boolean
     */
    public static boolean uploadOss(File file, String objectName) {
        return uploadOss(file, objectName, null, null);
    }

    /**
     * 上传Oss文件
     * 
     * @param file
     * @param objectName
     * @param contentType
     * @return boolean
     */
    public static boolean uploadOss(File file, String objectName, String contentType) {
        return uploadOss(file, objectName, contentType, null);
    }

    /**
     * 上传Oss文件
     * 
     * @param file
     * @param objectName
     * @param contentType
     * @param targetBucketName
     * @return
     */
    private static boolean uploadOss(File file, String objectName, String contentType, String targetBucketName) {

        // 获取Bucket的存在信息
        boolean resultFlag = false;

        String curreyBucketName = bucketName;
        if (StringUtils.isNotBlank(targetBucketName)) {
            curreyBucketName = targetBucketName;
        }
        if (!ossClient.doesBucketExist(curreyBucketName)) {
            AppLogUtil.warn(LOG, "oss上传失败，bucket【{}】不存在", curreyBucketName);
            return resultFlag;
        }
        try (InputStream content = new FileInputStream(file)) {

            // 创建上传Object的Metadata
            ObjectMetadata meta = new ObjectMetadata();
            // 必须设置ContentLength
            meta.setContentLength(file.length());
            if (!StringUtils.isEmpty(contentType)) {
                meta.setContentType(contentType);
            }
            boolean flag = false;
            if (isFileExist(objectName, curreyBucketName)) {
                flag = true;
            }
            // 上传Object.
            ossClient.putObject(curreyBucketName, objectName, content, meta);
            if (flag) {
                refreshObjectCaches(objectName);
            }
            resultFlag = true;
        } catch (OSSException | ClientException | IOException e) {
            LOG.error(e.getMessage(), e);
            resultFlag = false;
        }
        return resultFlag;
    }

    /**
     * 用来上传给服务器读取的配置文件
     * 
     * @param file
     * @param objectName
     * @param contentType
     * @return 访问地址
     */
    public static String uploadOssNotCDN(File file, String objectName, String contentType) {

        try {
            // 创建上传Object的Metadata
            ObjectMetadata meta = new ObjectMetadata();
            // 必须设置ContentLength
            meta.setContentLength(file.length());

            if (!StringUtils.isEmpty(contentType)) {
                meta.setContentType(contentType);
            }
            meta.setContentDisposition("Attachment;Filename=\"" + file.getName() + "\"");
            // 上传Object.
            ossClient.putObject(bucketName, objectName, file, meta);
            return objectPath + objectName;
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
            throw new QihoManagerException("上传cdn失败", e);
        }
    }

    /**
     * 上传OssCss文件
     * 
     * @param file
     * @param objectName
     * @return boolean
     */
    public static boolean uploadOssCss(File file, String objectName) {
        return uploadOss(file, objectName, "text/css");
    }

    /**
     * 上传OssJs文件
     * 
     * @param file
     * @param objectName
     * @return boolean
     */
    public static boolean uploadOssJs(File file, String objectName) {
        return uploadOss(file, objectName, "application/x-javascript");
    }

    /**
     * 上传OssHtml模板文件
     * 
     * @param file
     * @param objectName
     * @return boolean
     */
    public static boolean uploadOssHtml(File file, String objectName) {
        return uploadOss(file, objectName, "text/html");
    }

    /**
     * 上传OssImg文件
     * 
     * @param file
     * @param objectName
     * @return boolean
     */
    public static boolean uploadOssImg(File file, String objectName) {
        return uploadOss(file, objectName, "image/jpeg");
    }

    /**
     * 上传OssImg文件
     *
     * @param file
     * @param objectName
     * @return boolean
     */
    public static boolean uploadOssMp4(File file, String objectName) {
        return uploadOss(file, objectName, "video/mpeg4");
    }

    /**
     * CDN刷新缓存
     * 
     * @param path
     * @return boolean
     */
    public static boolean refreshObjectCaches(String path) {
        RefreshObjectCachesRequest request = new RefreshObjectCachesRequest();
        request.setObjectType(ObjectType.File.name());
        request.setObjectPath(objectPath + path);
        boolean resultFlag = false;
        try {
            RefreshObjectCachesResponse response = cdnClient.execute(request);
            if (StringUtils.isEmpty(response.getErrorCode())) {
                resultFlag = true;
            } else {
                // 刷新失败
                resultFlag = false;
            }
        } catch (ApiException e) {
            LOG.error(e.getMessage(), e);
            resultFlag = false;
        }
        return resultFlag;
    }

    /**
     * 验证文件是否存在
     * 
     * @param objectName
     * @return
     */
    private static boolean isFileExist(String objectName, String targetBucketName) {
        try {
            String curreyBucketName = bucketName;
            if (StringUtils.isNotBlank(targetBucketName)) {
                curreyBucketName = targetBucketName;
            }
            ossClient.getObjectMetadata(curreyBucketName, objectName);
            return true;
        } catch (ClientException e) {
            LOG.error(e.getMessage(), e);
            return false;
        } catch (OSSException e) {
            if (!"NoSuchKey".equals(e.getErrorCode())) {
                LOG.error(e.getMessage(), e);
            }
            return false;
        }
    }

    /**
     * 验证文件是否存在
     * 
     * @param objectName
     * @return boolean
     */
    public static boolean isFileExist(String objectName) {
        return isFileExist(objectName, null);
    }

    /**
     * 刷新路径类型
     */
    enum ObjectType {
        /* 文件 */
        File,
        /* 目录 */
        Directory;
    }

    /**
     *
     * @param objectName
     */
    public void deleteObject(String objectName) {
        try {
            ossClient.deleteObject(bucketName, objectName);
        } catch (OSSException | ClientException e) {
            LOG.error(e.getMessage(), e);
        }
    }

    /**
     * 上传文件,该文件会强制保存到yun.duiba.com.cn所在budget下,目前只有提供手动上传图片、js、css等资源的地方才应该调用此方法
     *
     * @param file
     * @param objectName
     * @return http url,如果上传失败则为空
     */
    public static String uploadAdminOssFile(File file, String objectName) {
        boolean resultOss = false;
        String fileType = objectName.substring(objectName.lastIndexOf('.') + 1, objectName.length());
        if (fileType.equalsIgnoreCase("jpg") || fileType.equalsIgnoreCase("jpeg") || fileType.equalsIgnoreCase("png")
            || fileType.equalsIgnoreCase("gif")) {
            resultOss = UploadTool.uploadOss(file, objectName, "image/jpeg", "duiba");
        } else if (fileType.equalsIgnoreCase("js")) {
            resultOss = UploadTool.uploadOss(file, objectName, "application/x-javascript", "duiba");
        } else if (fileType.equalsIgnoreCase("css")) {
            resultOss = UploadTool.uploadOss(file, objectName, "text/css", "duiba");
        } else if (fileType.equalsIgnoreCase("html")) {
            resultOss = UploadTool.uploadOss(file, objectName, "text/html", "duiba");
        } else {
            resultOss = UploadTool.uploadOss(file, objectName, null, "duiba");
        }

        if (resultOss) {
            return "https://" + objectPath + "/" + objectName;
        } else {
            return null;
        }
    }

    public static void setBucketName(String bucketName) {
        UploadTool.bucketName = bucketName;
    }

    public static void setObjectPath(String objectPath) {
        UploadTool.objectPath = objectPath;
    }

    public static void setOssClient(OSSClient ossClient) {
        UploadTool.ossClient = ossClient;
    }

    public static void setCdnClient(AliyunClient cdnClient) {
        UploadTool.cdnClient = cdnClient;
    }

}
