package com.qiho.manager.common.util;

import cn.com.duiba.wolf.utils.BlowfishUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.regex.Pattern;

/**
 * 订单相关的工具类
 *
 * @author chensong
 * @create 2018-09-14
 */
public class BaiqiOrderUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(BaiqiOrderUtil.class);

    /** 订单id正则表达式 */
    private static final Pattern ORDER_ID_PATTERN = Pattern.compile("^[0-9]{19}[D][0-9]{8}$");

    /** orderId token key */
    private static final String TOKEN_KEY = "token";

    private BaiqiOrderUtil(){}

    /***
     * 根据orderId加密生成orderId的token
     * 当输入字符串不是orderId时或者生成出错将返回null
     *
     * @param orderId 订单id
     * @return token
     */
    public static String generateOrderToken(String orderId){
        if (!matchOrderId(orderId)) {
            return null;
        }

        try{
            return BlowfishUtils.encryptBlowfish(orderId, TOKEN_KEY);
        } catch (Exception e){
            LOGGER.error("生成orderToken出错，orderId={}", orderId, e);
            return null;
        }
    }


    /**
     * 将token解析为orderId
     *
     * @param token
     * @return
     */
    public static String parsingOrderToken(String token){
        if (StringUtils.isBlank(token)) {
            return null;
        }

        String orderId;
        try{
            orderId = BlowfishUtils.decryptBlowfish(token, TOKEN_KEY);
        } catch (Exception e){
            LOGGER.error("解析order token异常，token={}", token, e);
            return null;
        }

        if (!matchOrderId(orderId)) {
            LOGGER.warn("解析order token错误，token={}", token);
            return null;
        }

        return orderId;
    }


    /**
     * 匹配输入字符串是否是orderId形式
     * @param value
     * @return true-是  false-不是
     */
    public static boolean matchOrderId(String value){
        if (StringUtils.isBlank(value)) {
            return false;
        }
        return ORDER_ID_PATTERN.matcher(value).matches();
    }



}
