/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.memory;

import java.nio.ByteBuffer;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.flink.core.memory.HybridMemorySegment;
import org.apache.flink.core.memory.MemorySegment;
import org.apache.flink.core.memory.MemorySegmentFactory;
import org.apache.flink.core.memory.MemoryType;
import org.apache.flink.runtime.memory.MemoryAllocationException;
import org.apache.flink.util.MathUtils;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MemoryManager {
    private static final Logger LOG = LoggerFactory.getLogger(MemoryManager.class);
    public static final int DEFAULT_PAGE_SIZE = 32768;
    public static final int MIN_PAGE_SIZE = 4096;
    private final Object lock = new Object();
    private final MemoryPool memoryPool;
    private final AllocatedSegmentsWrapper allocatedCoreSegments;
    private final AllocatedSegmentsWrapper allocatedFloatingSegments;
    private final int pageSize;
    private final int totalNumPages;
    private final long coreMemorySize;
    private final long floatingMemorySize;
    private final int numberOfSlots;
    private final boolean isPreAllocated;
    private boolean isShutDown;

    public MemoryManager(long coreMemorySize, int numberOfSlots) {
        this(coreMemorySize, 0L, numberOfSlots, 32768, MemoryType.HEAP, true);
    }

    public MemoryManager(long coreMemorySize, int numberOfSlots, int pageSize, MemoryType memoryType, boolean preAllocateMemory) {
        this(coreMemorySize, 0L, numberOfSlots, pageSize, memoryType, preAllocateMemory);
    }

    public MemoryManager(long coreMemorySize, long floatingMemorySize, int numberOfSlots, int pageSize, MemoryType memoryType, boolean preAllocateMemory) {
        if (memoryType == null) {
            throw new NullPointerException();
        }
        if (coreMemorySize <= 0L) {
            throw new IllegalArgumentException("Size of total core memory must be positive.");
        }
        if (floatingMemorySize < 0L) {
            throw new IllegalArgumentException("Size of total floating memory must not be negative.");
        }
        if (pageSize < 4096) {
            throw new IllegalArgumentException("The page size must be at least 4096 bytes.");
        }
        if (!MathUtils.isPowerOf2((long)pageSize)) {
            throw new IllegalArgumentException("The given page size is not a power of two.");
        }
        this.coreMemorySize = coreMemorySize;
        this.floatingMemorySize = floatingMemorySize;
        this.numberOfSlots = numberOfSlots;
        this.pageSize = pageSize;
        int numCorePages = this.calculateNumPages(coreMemorySize);
        int numFloatingPages = floatingMemorySize > 0L ? this.calculateNumPages(floatingMemorySize) : 0;
        this.totalNumPages = numCorePages + numFloatingPages;
        this.allocatedCoreSegments = new AllocatedSegmentsWrapper(numCorePages);
        this.allocatedFloatingSegments = new AllocatedSegmentsWrapper(numFloatingPages);
        this.isPreAllocated = preAllocateMemory;
        int memToAllocate = preAllocateMemory ? this.totalNumPages : 0;
        switch (memoryType) {
            case HEAP: {
                this.memoryPool = new HybridHeapMemoryPool(memToAllocate, pageSize);
                break;
            }
            case OFF_HEAP: {
                if (!preAllocateMemory) {
                    LOG.warn("It is advisable to set 'taskmanager.memory.preallocate' to true when the memory type 'taskmanager.memory.off-heap' is set to true.");
                }
                this.memoryPool = new HybridOffHeapMemoryPool(memToAllocate, pageSize);
                break;
            }
            default: {
                throw new IllegalArgumentException("unrecognized memory type: " + memoryType);
            }
        }
    }

    private int calculateNumPages(long memorySize) {
        long numPagesLong = memorySize / (long)this.pageSize;
        if (numPagesLong > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("The given number of memory bytes (" + memorySize + ") corresponds to more than MAX_INT pages.");
        }
        int totalNumPages = (int)numPagesLong;
        if (totalNumPages < 1) {
            throw new IllegalArgumentException("The given amount of core memory amounted to less than one page.");
        }
        return totalNumPages;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() {
        Object object = this.lock;
        synchronized (object) {
            if (!this.isShutDown) {
                this.isShutDown = true;
                this.allocatedCoreSegments.freeAllocatedSegments();
                this.allocatedFloatingSegments.freeAllocatedSegments();
                this.memoryPool.clear();
            }
        }
    }

    public boolean isShutdown() {
        return this.isShutDown;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean verifyEmpty() {
        Object object = this.lock;
        synchronized (object) {
            int numTotalAvailablePages = this.allocatedCoreSegments.getNumAvailablePages() + this.allocatedFloatingSegments.getNumAvailablePages();
            return this.isPreAllocated ? this.memoryPool.getNumberOfAvailableMemorySegments() == this.totalNumPages && numTotalAvailablePages == this.totalNumPages : numTotalAvailablePages == this.totalNumPages;
        }
    }

    public List<MemorySegment> allocatePages(Object owner, int numPages) throws MemoryAllocationException {
        ArrayList<MemorySegment> segments = new ArrayList<MemorySegment>(numPages);
        this.allocatePages(owner, segments, numPages);
        return segments;
    }

    public void allocatePages(Object owner, List<MemorySegment> target, int numPages) throws MemoryAllocationException {
        this.allocatePages(owner, target, numPages, true);
    }

    public List<MemorySegment> allocatePages(Object owner, int numPages, boolean isCoreSegment) throws MemoryAllocationException {
        ArrayList<MemorySegment> segments = new ArrayList<MemorySegment>(numPages);
        this.allocatePages(owner, segments, numPages, isCoreSegment);
        return segments;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void allocatePages(Object owner, List<MemorySegment> target, int numPages, boolean isCoreSegment) throws MemoryAllocationException {
        if (owner == null) {
            throw new IllegalArgumentException("The memory owner must not be null.");
        }
        if (target instanceof ArrayList) {
            ((ArrayList)target).ensureCapacity(numPages);
        }
        Object object = this.lock;
        synchronized (object) {
            if (this.isShutDown) {
                throw new IllegalStateException("Memory manager has been shut down.");
            }
            AllocatedSegmentsWrapper allocatedSegments = isCoreSegment ? this.allocatedCoreSegments : this.allocatedFloatingSegments;
            Set<MemorySegment> segmentsForOwner = allocatedSegments.allocateSegments(owner, numPages);
            if (this.isPreAllocated) {
                for (int i = numPages; i > 0; --i) {
                    MemorySegment segment = this.memoryPool.requestSegmentFromPool(owner);
                    target.add(segment);
                    segmentsForOwner.add(segment);
                }
            } else {
                for (int i = numPages; i > 0; --i) {
                    MemorySegment segment = this.memoryPool.allocateNewSegment(owner);
                    target.add(segment);
                    segmentsForOwner.add(segment);
                }
            }
        }
    }

    public void release(@Nullable MemorySegment segment) {
        this.release(segment, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean release(@Nullable MemorySegment segment, boolean isCoreSegment) {
        if (segment == null || segment.getOwner() == null) {
            return false;
        }
        Object object = this.lock;
        synchronized (object) {
            if (segment.isFreed()) {
                return false;
            }
            if (this.isShutDown) {
                throw new IllegalStateException("Memory manager has been shut down.");
            }
            AllocatedSegmentsWrapper allocatedSegments = isCoreSegment ? this.allocatedCoreSegments : this.allocatedFloatingSegments;
            boolean hasRemoved = allocatedSegments.removeAllocatedSegments(segment);
            if (hasRemoved) {
                this.releaseMemorySegment(segment);
            }
            return hasRemoved;
        }
    }

    public void release(@Nullable Collection<MemorySegment> segments) {
        this.release(segments, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void release(@Nullable Collection<MemorySegment> segments, boolean isCoreSegment) {
        if (segments == null) {
            return;
        }
        ArrayList<MemorySegment> notReleased = new ArrayList<MemorySegment>();
        Object object = this.lock;
        synchronized (object) {
            if (this.isShutDown) {
                throw new IllegalStateException("Memory manager has been shut down.");
            }
            boolean successfullyReleased = false;
            do {
                Iterator<MemorySegment> segmentsIterator = segments.iterator();
                AllocatedSegmentsWrapper allocatedSegments = isCoreSegment ? this.allocatedCoreSegments : this.allocatedFloatingSegments;
                try {
                    while (segmentsIterator.hasNext()) {
                        MemorySegment seg = segmentsIterator.next();
                        if (seg == null || seg.isFreed()) continue;
                        boolean hasRemoved = allocatedSegments.removeAllocatedSegments(seg);
                        if (hasRemoved) {
                            this.releaseMemorySegment(seg);
                            continue;
                        }
                        notReleased.add(seg);
                    }
                    segments.clear();
                    segments.addAll(notReleased);
                    successfullyReleased = true;
                }
                catch (ConcurrentModificationException concurrentModificationException) {
                    // empty catch block
                }
            } while (!successfullyReleased);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void releaseAll(Object owner) {
        if (owner == null) {
            return;
        }
        Object object = this.lock;
        synchronized (object) {
            if (this.isShutDown) {
                throw new IllegalStateException("Memory manager has been shut down.");
            }
            this.releaseMemorySegments(this.allocatedCoreSegments.removeAllocatedSegments(owner));
            this.releaseMemorySegments(this.allocatedFloatingSegments.removeAllocatedSegments(owner));
        }
    }

    private void releaseMemorySegments(Set<MemorySegment> segments) {
        if (segments == null || segments.isEmpty()) {
            return;
        }
        if (this.isPreAllocated) {
            for (MemorySegment seg : segments) {
                this.memoryPool.returnSegmentToPool(seg);
            }
        } else {
            for (MemorySegment seg : segments) {
                seg.free();
            }
        }
        segments.clear();
    }

    private void releaseMemorySegment(MemorySegment seg) {
        if (this.isPreAllocated) {
            this.memoryPool.returnSegmentToPool(seg);
        } else {
            seg.free();
        }
    }

    public int getPageSize() {
        return this.pageSize;
    }

    public long getMemorySize() {
        return this.coreMemorySize + this.floatingMemorySize;
    }

    public int computeNumberOfPages(double fraction) {
        if (fraction <= 0.0 || fraction > 1.0) {
            throw new IllegalArgumentException("The fraction of memory to allocate must within (0, 1].");
        }
        return (int)((double)this.totalNumPages * fraction / (double)this.numberOfSlots);
    }

    static final class HybridOffHeapMemoryPool
    extends MemoryPool {
        private final ArrayDeque<ByteBuffer> availableMemory;
        private final int segmentSize;

        HybridOffHeapMemoryPool(int numInitialSegments, int segmentSize) {
            this.availableMemory = new ArrayDeque(numInitialSegments);
            this.segmentSize = segmentSize;
            for (int i = 0; i < numInitialSegments; ++i) {
                this.availableMemory.add(ByteBuffer.allocateDirect(segmentSize));
            }
        }

        @Override
        MemorySegment allocateNewSegment(Object owner) {
            ByteBuffer memory = ByteBuffer.allocateDirect(this.segmentSize);
            return MemorySegmentFactory.wrapPooledOffHeapMemory((ByteBuffer)memory, (Object)owner);
        }

        @Override
        MemorySegment requestSegmentFromPool(Object owner) {
            ByteBuffer buf = this.availableMemory.remove();
            return MemorySegmentFactory.wrapPooledOffHeapMemory((ByteBuffer)buf, (Object)owner);
        }

        @Override
        void returnSegmentToPool(MemorySegment segment) {
            if (segment.getClass() != HybridMemorySegment.class) {
                throw new IllegalArgumentException("Memory segment is not a " + HybridMemorySegment.class.getSimpleName());
            }
            HybridMemorySegment hybridSegment = (HybridMemorySegment)segment;
            ByteBuffer buf = hybridSegment.getOffHeapBuffer();
            this.availableMemory.add(buf);
            hybridSegment.free();
        }

        @Override
        protected int getNumberOfAvailableMemorySegments() {
            return this.availableMemory.size();
        }

        @Override
        void clear() {
            this.availableMemory.clear();
        }
    }

    static final class HybridHeapMemoryPool
    extends MemoryPool {
        private final ArrayDeque<byte[]> availableMemory;
        private final int segmentSize;

        HybridHeapMemoryPool(int numInitialSegments, int segmentSize) {
            this.availableMemory = new ArrayDeque(numInitialSegments);
            this.segmentSize = segmentSize;
            for (int i = 0; i < numInitialSegments; ++i) {
                this.availableMemory.add(new byte[segmentSize]);
            }
        }

        @Override
        MemorySegment allocateNewSegment(Object owner) {
            return MemorySegmentFactory.allocateUnpooledSegment((int)this.segmentSize, (Object)owner);
        }

        @Override
        MemorySegment requestSegmentFromPool(Object owner) {
            byte[] buf = this.availableMemory.remove();
            return MemorySegmentFactory.wrapPooledHeapMemory((byte[])buf, (Object)owner);
        }

        @Override
        void returnSegmentToPool(MemorySegment segment) {
            if (segment.getClass() != HybridMemorySegment.class) {
                throw new IllegalArgumentException("Memory segment is not a " + HybridMemorySegment.class.getSimpleName());
            }
            HybridMemorySegment heapSegment = (HybridMemorySegment)segment;
            this.availableMemory.add(heapSegment.getArray());
            heapSegment.free();
        }

        @Override
        protected int getNumberOfAvailableMemorySegments() {
            return this.availableMemory.size();
        }

        @Override
        void clear() {
            this.availableMemory.clear();
        }
    }

    static abstract class MemoryPool {
        MemoryPool() {
        }

        abstract int getNumberOfAvailableMemorySegments();

        abstract MemorySegment allocateNewSegment(Object var1);

        abstract MemorySegment requestSegmentFromPool(Object var1);

        abstract void returnSegmentToPool(MemorySegment var1);

        abstract void clear();
    }

    private static class AllocatedSegmentsWrapper {
        private final HashMap<Object, Set<MemorySegment>> allocatedSegments;
        private int numAvailablePages;

        AllocatedSegmentsWrapper(int numPages) {
            this.numAvailablePages = numPages;
            this.allocatedSegments = new HashMap();
        }

        @Nonnull
        Set<MemorySegment> allocateSegments(Object owner, int numRequiredPages) throws MemoryAllocationException {
            if (numRequiredPages > this.numAvailablePages) {
                throw new MemoryAllocationException("Could not allocate " + numRequiredPages + " pages. Only " + this.numAvailablePages + " pages are remaining.");
            }
            Set<MemorySegment> segmentsForOwner = this.allocatedSegments.get(owner);
            if (segmentsForOwner == null) {
                segmentsForOwner = new HashSet<MemorySegment>(numRequiredPages);
                this.allocatedSegments.put(owner, segmentsForOwner);
            }
            this.numAvailablePages -= numRequiredPages;
            return segmentsForOwner;
        }

        boolean removeAllocatedSegments(MemorySegment segment) {
            Object owner = Preconditions.checkNotNull((Object)segment.getOwner());
            Set<MemorySegment> segmentsForOwner = this.allocatedSegments.get(owner);
            if (segmentsForOwner != null) {
                boolean hasRemoved = segmentsForOwner.remove(segment);
                if (hasRemoved) {
                    if (segmentsForOwner.isEmpty()) {
                        this.allocatedSegments.remove(owner);
                    }
                    ++this.numAvailablePages;
                }
                return hasRemoved;
            }
            return false;
        }

        @Nullable
        Set<MemorySegment> removeAllocatedSegments(Object owner) {
            Set<MemorySegment> segmentsForOwner = this.allocatedSegments.remove(owner);
            if (segmentsForOwner != null) {
                this.numAvailablePages += segmentsForOwner.size();
            }
            return segmentsForOwner;
        }

        void freeAllocatedSegments() {
            for (Set<MemorySegment> segments : this.allocatedSegments.values()) {
                for (MemorySegment seg : segments) {
                    seg.free();
                }
            }
            this.allocatedSegments.clear();
        }

        int getNumAvailablePages() {
            return this.numAvailablePages;
        }
    }
}

