/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.calcite.type;

import org.apache.calcite.sql.SqlCallBinding;
import org.apache.calcite.sql.SqlNode;
import org.apache.calcite.sql.SqlOperandCountRange;
import org.apache.calcite.sql.SqlOperator;
import org.apache.calcite.sql.type.OperandTypes;
import org.apache.calcite.sql.type.SqlSingleOperandTypeChecker;
import org.apache.calcite.sql.type.SqlTypeFamily;

import java.util.List;

/**
 * Single operand type checker that checks operand at `position` must be specific Literal values.
 */
public class SpecificLiteralOperandTypeChecker implements SqlSingleOperandTypeChecker {

	private final List<Object> literalVals;
	private final int position;
	private final SqlSingleOperandTypeChecker innerChecker;

	public SpecificLiteralOperandTypeChecker(
		SqlTypeFamily family, int position, List literalVals) {
		if (literalVals.isEmpty()) {
			throw new IllegalArgumentException("Possible literal values must be provided.");
		}
		this.position = position;
		this.innerChecker = OperandTypes.family(family);
		this.literalVals = literalVals;
	}

	@Override
	public boolean checkOperandTypes(SqlCallBinding callBinding, boolean throwOnFailure) {
		return innerChecker.checkOperandTypes(callBinding, throwOnFailure);
	}

	@Override
	public SqlOperandCountRange getOperandCountRange() {
		return innerChecker.getOperandCountRange();
	}

	@Override
	public String getAllowedSignatures(SqlOperator op, String opName) {
		return innerChecker.getAllowedSignatures(op, opName) +
			", and the value must be in " + literalVals;
	}

	@Override
	public Consistency getConsistency() {
		return innerChecker.getConsistency();
	}

	@Override
	public boolean isOptional(int i) {
		return false;
	}

	@Override
	public boolean checkSingleOperandType(
		SqlCallBinding callBinding,
		SqlNode operand,
		int iFormalOperand,
		boolean throwOnFailure) {

		Object value = callBinding.getOperandLiteralValue(position, literalVals.get(0).getClass());
		return value != null && literalVals.contains(value);
	}
}
