/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.plan.nodes.calcite

import org.apache.flink.table.calcite.FlinkRelBuilder.NamedWindowProperty
import org.apache.flink.table.plan.logical.LogicalWindow

import org.apache.calcite.plan.{Convention, RelOptCluster, RelTraitSet}
import org.apache.calcite.rel.RelNode
import org.apache.calcite.rel.core.Aggregate.Group
import org.apache.calcite.rel.core.{Aggregate, AggregateCall}
import org.apache.calcite.util.ImmutableBitSet

import scala.collection.JavaConverters._

final class LogicalWindowAggregate(
    window: LogicalWindow,
    namedProperties: Seq[NamedWindowProperty],
    cluster: RelOptCluster,
    traitSet: RelTraitSet,
    child: RelNode,
    groupSet: ImmutableBitSet,
    aggCalls: Seq[AggregateCall])
  extends WindowAggregate(
    window,
    namedProperties,
    cluster,
    traitSet,
    child,
    groupSet,
    aggCalls) {

  def copy(namedProperties: Seq[NamedWindowProperty]): WindowAggregate = {
    new LogicalWindowAggregate(
      window,
      namedProperties,
      cluster,
      traitSet,
      input,
      groupSet,
      aggCalls)
  }

  override def copy(
      traitSet: RelTraitSet,
      input: RelNode,
      aggCalls: Seq[AggregateCall]): WindowAggregate = {
    new LogicalWindowAggregate(
      window,
      namedProperties,
      cluster,
      traitSet,
      input,
      groupSet,
      aggCalls)
  }
}

object LogicalWindowAggregate {
  def create(
      window: LogicalWindow,
      namedProperties: Seq[NamedWindowProperty],
      agg: Aggregate): WindowAggregate = {
    // GROUPING SETS are currently not supported.
    require(!agg.indicator && (agg.getGroupType == Group.SIMPLE))
    val cluster: RelOptCluster = agg.getCluster
    val traitSet: RelTraitSet = cluster.traitSetOf(Convention.NONE)

    new LogicalWindowAggregate(
      window,
      namedProperties,
      cluster,
      traitSet,
      agg.getInput(),
      agg.getGroupSet,
      agg.getAggCallList.asScala)
  }
}
