/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.plan.nodes.logical

import org.apache.flink.table.plan.metadata.FlinkRelMetadataQuery
import org.apache.flink.table.plan.nodes.FlinkConventions

import com.google.common.collect.ImmutableList
import org.apache.calcite.plan._
import org.apache.calcite.rel.`type`.RelDataType
import org.apache.calcite.rel.convert.ConverterRule
import org.apache.calcite.rel.core.Values
import org.apache.calcite.rel.logical.LogicalValues
import org.apache.calcite.rel.metadata.{RelMdCollation, RelMetadataQuery}
import org.apache.calcite.rel.{RelCollation, RelCollationTraitDef, RelNode}
import org.apache.calcite.rex.RexLiteral

import java.util
import java.util.function.Supplier

class FlinkLogicalValues(
    cluster: RelOptCluster,
    traitSet: RelTraitSet,
    rowRelDataType: RelDataType,
    tuples: ImmutableList[ImmutableList[RexLiteral]])
  extends Values(cluster, rowRelDataType, tuples, traitSet)
  with FlinkLogicalRel {

  override def copy(traitSet: RelTraitSet, inputs: java.util.List[RelNode]): RelNode = {
    new FlinkLogicalValues(cluster, traitSet, rowRelDataType, tuples)
  }

  override def computeSelfCost(planner: RelOptPlanner, mq: RelMetadataQuery): RelOptCost = {
    val dRows = mq.getRowCount(this)
    // Assume CPU is negligible since values are precomputed.
    val dCpu = 1
    val dIo = 0
    planner.getCostFactory.makeCost(dRows, dCpu, dIo)
  }

  override def isDeterministic: Boolean = true
}

private class FlinkLogicalValuesConverter
  extends ConverterRule(
    classOf[LogicalValues],
    Convention.NONE,
    FlinkConventions.LOGICAL,
    "FlinkLogicalValuesConverter") {

  override def convert(rel: RelNode): RelNode = {
    val values = rel.asInstanceOf[LogicalValues]
    FlinkLogicalValues.create(rel.getCluster, values.getRowType, values.getTuples())
  }
}

object FlinkLogicalValues {

  val CONVERTER: ConverterRule = new FlinkLogicalValuesConverter()

  def create(
      cluster: RelOptCluster,
      rowType: RelDataType,
      tuples: ImmutableList[ImmutableList[RexLiteral]]): FlinkLogicalValues = {
    val mq = cluster.getMetadataQuery
    val traitSet = cluster.traitSetOf(Convention.NONE).replaceIfs(
      RelCollationTraitDef.INSTANCE, new Supplier[util.List[RelCollation]]() {
        def get: util.List[RelCollation] = RelMdCollation.values(mq, rowType, tuples)
      })
    // FIXME: FlinkRelMdDistribution requires the current RelNode to compute
    // the distribution trait, so we have to create FlinkLogicalValues to
    // calculate the distribution trait
    val values = new FlinkLogicalValues(cluster, traitSet, rowType, tuples)
    val newTraitSet = FlinkRelMetadataQuery.traitSet(values)
      .replace(FlinkConventions.LOGICAL).simplify()
    values.copy(newTraitSet, values.getInputs).asInstanceOf[FlinkLogicalValues]
  }
}
