/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.plan.nodes.physical.stream

import org.apache.flink.streaming.api.transformations.{OneInputTransformation, StreamTransformation}
import org.apache.flink.table.api.window.TimeWindow
import org.apache.flink.table.api.{StreamTableEnvironment, TableConfig, TableConfigOptions, TableException}
import org.apache.flink.table.calcite.FlinkRelBuilder.NamedWindowProperty
import org.apache.flink.table.codegen.GeneratedSubKeyedAggsHandleFunction
import org.apache.flink.table.dataformat.BaseRow
import org.apache.flink.table.errorcode.TableErrors
import org.apache.flink.table.expressions.ExpressionUtils._
import org.apache.flink.table.plan.logical.{LogicalWindow, SlidingGroupWindow, TumblingGroupWindow}
import org.apache.flink.table.plan.nodes.exec.{ExecNodeWriter, RowStreamExecNode}
import org.apache.flink.table.plan.nodes.physical.FlinkPhysicalRel
import org.apache.flink.table.plan.rules.physical.stream.StreamExecRetractionRules
import org.apache.flink.table.plan.schema.BaseRowSchema
import org.apache.flink.table.plan.util.{AggregateInfoList, AggregateNameUtil, AggregateUtil, StreamExecUtil, WindowAggregateUtil}
import org.apache.flink.table.runtime.fault.tolerant.FaultTolerantUtil
import org.apache.flink.table.runtime.window.aligned.{InternalAlignedWindowTriggers, LocalAlignedWindowAggregator}
import org.apache.flink.table.runtime.window.assigners.{SlidingWindowAssigner, TumblingWindowAssigner}
import org.apache.flink.table.runtime.window.{AbstractAlignedWindowOperator, LocalAlignedWindowOperator}
import org.apache.flink.table.types.{DataTypes, InternalType, TypeConverters}
import org.apache.flink.table.typeutils.BaseRowTypeInfo
import org.apache.flink.util.Preconditions

import org.apache.calcite.plan.{RelOptCluster, RelTraitSet}
import org.apache.calcite.rel.`type`.RelDataType
import org.apache.calcite.rel.core.AggregateCall
import org.apache.calcite.rel.{RelNode, RelWriter, SingleRel}

import scala.collection.JavaConverters._

/**
  * Flink RelNode for stream local group window aggregate.
  */
class StreamExecLocalWindowAggregate(
    val window: LogicalWindow,
    cluster: RelOptCluster,
    traitSet: RelTraitSet,
    inputNode: RelNode,
    val aggInfoList: AggregateInfoList,
    val aggCalls: Seq[AggregateCall],
    outputSchema: BaseRowSchema,
    inputSchema: BaseRowSchema,
    grouping: Array[Int],
    inputTimestampIndex: Int)
  extends SingleRel(cluster, traitSet, inputNode)
  with StreamPhysicalRel
  with RowStreamExecNode {

  override def isDeterministic: Boolean = AggregateUtil.isDeterministic(aggCalls.asJava)

  override def deriveRowType(): RelDataType = outputSchema.relDataType

  // do not support early/late fire now.
  override def producesUpdates: Boolean = false

  override def consumesRetractions: Boolean = true

  // only support rowtime window in Local-Global now.
  override def requireWatermark: Boolean = true

  def getGroupings: Array[Int] = grouping

  override def copy(traitSet: RelTraitSet, inputs: java.util.List[RelNode]): RelNode = {
    new StreamExecLocalWindowAggregate(
      window,
      cluster,
      traitSet,
      inputs.get(0),
      aggInfoList,
      aggCalls,
      outputSchema,
      inputSchema,
      grouping,
      inputTimestampIndex)
  }

  override def explainTerms(pw: RelWriter): RelWriter = {
    super.explainTerms(pw)
      .itemIf("groupBy",
        AggregateNameUtil.groupingToString(inputSchema.relDataType, grouping), grouping.nonEmpty)
      .item("window", window)
      .item(
        "select", AggregateNameUtil.aggregationToString(
          inputSchema.relDataType,
          grouping,
          outputSchema.relDataType,
          aggCalls,
          Array.empty[NamedWindowProperty],
          withOutputFieldNames = true,
          isLocal = true))
  }

  /**
    * Describes the state digest of the ExecNode which is used to set the
    * user-specified ID of the translated stream operator. The user-specified ID
    * is used to assign the same operator ID across job restarts.
    * It is important for the mapping of operator state to the operator.
    *
    * Note: Be careful to modify this method as it will affect the state reuse.
    */
  override def getStateDigest(pw: ExecNodeWriter): ExecNodeWriter = {
    pw.item("inputType", input.getRowType)
      .itemIf("groupBy",
        AggregateNameUtil.groupingToString(inputSchema.relDataType, grouping), grouping.nonEmpty)
      .item("window", window)
      .item("select",
        AggregateNameUtil.aggregationToString(
          inputSchema.relDataType,
          grouping,
          outputSchema.relDataType,
          aggCalls,
          Array.empty[NamedWindowProperty],
          withOutputFieldNames = false,
          isLocal = true))
  }

  /**
    * Returns the [[FlinkPhysicalRel]] object corresponding to this node.
    */
  override def getFlinkPhysicalRel: FlinkPhysicalRel = this

  override protected def translateToPlanInternal(
    tableEnv: StreamTableEnvironment): StreamTransformation[BaseRow] = {

    val tableConfig = tableEnv.getConfig
    // only support rowtime window now.
    Preconditions.checkArgument(isRowtimeAttribute(window.timeAttribute))

    val inputTransform = getInputNodes.get(0).translateToPlan(tableEnv)
      .asInstanceOf[StreamTransformation[BaseRow]]
    val inputRowType = inputTransform.getOutputType.asInstanceOf[BaseRowTypeInfo]

    val needRetraction = StreamExecRetractionRules.isAccRetract(getInput)
    if (needRetraction) {
      // do not support retraction on windowed agg now.
      throw new TableException(
        TableErrors.INST.sqlGroupWindowAggTranslateRetractNotSupported())
    }

    val aggsHandler = WindowAggregateUtil.createAggsHandler(
      "LocalGroupWindowAggsHandler",
      window,
      Array.empty[NamedWindowProperty],
      aggInfoList,
      tableConfig,
      tableEnv.getRelBuilder,
      inputSchema.fieldTypes,
      needRetraction,
      true,
      0,
      // hint the merged acc is always on heap in local-window mode.
      true)

    val accTypes = aggInfoList.getAccTypes.map(_.toInternalType)
    val outRowType = outputSchema.typeInfo()

    val operator = createLocalWindowOperator(
      tableConfig,
      aggsHandler,
      inputRowType,
      accTypes,
      inputTimestampIndex)
    val operatorName = getOperatorName
    val transformation = new OneInputTransformation(
      inputTransform,
      operatorName,
      FaultTolerantUtil.addFaultTolerantProxyIfNeed(
        operator,
        operatorName,
        tableConfig),
      outRowType,
      inputTransform.getParallelism)

    transformation.setResources(
      getResource.getReservedResourceSpec,
      getResource.getPreferResourceSpec)
    transformation
  }

  private def createLocalWindowOperator(
    config: TableConfig,
    aggsHandler: GeneratedSubKeyedAggsHandleFunction[_],
    inputRowType: BaseRowTypeInfo,
    accTypes: Array[InternalType],
    timeIndex: Int): AbstractAlignedWindowOperator = {

    val keySelector = StreamExecUtil.getKeySelector(grouping, inputRowType)
    val accTypeInfo = TypeConverters.createInternalTypeInfoFromDataType(
      DataTypes.createRowType(accTypes: _*))
    val miniBatchSize = config.getConf.getLong(TableConfigOptions.SQL_EXEC_MINIBATCH_SIZE)

    val windowRunner = new LocalAlignedWindowAggregator(
      keySelector.getProducedType,
      accTypeInfo.asInstanceOf[BaseRowTypeInfo],
      aggsHandler.asInstanceOf[GeneratedSubKeyedAggsHandleFunction[TimeWindow]],
      miniBatchSize)

    // we should reverse the offset because assigner needed
    val (windowAssigner, windowTrigger) = window match {
      case TumblingGroupWindow(_, timeField, size)
        if isRowtimeAttribute(timeField) && isTimeIntervalLiteral(size) =>
        val sizeDuration = toDuration(size)
        val assigner = TumblingWindowAssigner.of(sizeDuration).withTimeZone(config.getTimeZone)
        val trigger = InternalAlignedWindowTriggers.tumbling(sizeDuration, config.getTimeZone)
        (assigner, trigger)
      case SlidingGroupWindow(_, timeField, size, slide)
        if isRowtimeAttribute(timeField) && isTimeIntervalLiteral(size) =>
        val sizeDuration = toDuration(size)
        val slideDuration = toDuration(slide)
        val assigner = SlidingWindowAssigner
          .of(sizeDuration, slideDuration)
          .withTimeZone(config.getTimeZone)
        val trigger = InternalAlignedWindowTriggers.sliding(
          sizeDuration, slideDuration, config.getTimeZone)
        (assigner, trigger)
    }

    val finishBundleBeforeSnapshot = config.getConf.getBoolean(
      TableConfigOptions.SQL_EXEC_MINIBATCH_FLUSH_BEFORE_SNAPSHOT)

    new LocalAlignedWindowOperator(
      windowRunner,
      windowAssigner,
      windowTrigger,
      timeIndex,
      finishBundleBeforeSnapshot,
      keySelector)
  }

  private def getOperatorName: String = {
    val windowAggString = AggregateNameUtil.aggregationToString(
      inputSchema.relDataType,
      grouping,
      outputSchema.relDataType,
      aggCalls,
      Array.empty[NamedWindowProperty],
      withOutputFieldNames = true,
      isLocal = true)

    val windowPrefix = "local-window"
    if (grouping.nonEmpty) {
      s"$windowPrefix: ($window), " +
        s"groupBy: (${AggregateNameUtil.groupingToString(inputSchema.relDataType, grouping)}), " +
        s"select: ($windowAggString)"
    } else {
      s"$windowPrefix: ($window), select: ($windowAggString)"
    }
  }

}
