/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.plan.rules.physical.batch.runtimefilter

import org.apache.flink.table.functions.sql.internal.SqlRuntimeFilterFunction
import org.apache.flink.table.plan.FlinkJoinRelType.{INNER, SEMI}
import org.apache.flink.table.plan.nodes.physical.batch.{BatchExecCalc, BatchExecHashJoinBase}
import org.apache.flink.table.plan.rules.physical.batch.runtimefilter.BaseRuntimeFilterPushDownRule.findRuntimeFilters
import org.apache.flink.table.plan.rules.physical.batch.runtimefilter.InsertRuntimeFilterRule.getHepRel
import org.apache.flink.table.plan.rules.physical.batch.runtimefilter.UselessRuntimeFilterRemoveRule._

import org.apache.calcite.plan.RelOptRule.{any, operand}
import org.apache.calcite.plan.{RelOptRule, RelOptRuleCall}
import org.apache.calcite.rel.RelNode
import org.apache.calcite.rex.RexInputRef
import org.apache.calcite.sql.SqlOperator

import scala.collection.JavaConversions._
import scala.collection.mutable

/**
  * Planner rule that removes a useless [[SqlRuntimeFilterFunction]] (Not worth doing) for
  * Broadcast join.
  */
class UselessBroadcastRfRemoveRule extends RelOptRule(
  operand(classOf[BatchExecHashJoinBase], operand(classOf[RelNode], any)),
  "UselessBroadcastRfRemoveRule") {

  override def matches(call: RelOptRuleCall): Boolean = {
    val join: BatchExecHashJoinBase = call.rel(0)

    join.isBroadcast &&
        (join.flinkJoinType == INNER || join.flinkJoinType == SEMI) &&
        join.haveInsertRf
  }

  override def onMatch(call: RelOptRuleCall): Unit = {
    val join: BatchExecHashJoinBase = call.rel(0)
    getHepRel(join.probeRel) match {
      case calc: BatchExecCalc =>
        val rfCalls = findRuntimeFilters(calc.getProgram)
        val toRemove = new mutable.ArrayBuffer[SqlOperator]

        val calcFields = calc.getProgram.getProjectList.map(calc.getProgram.expandLocalRef)
        val probeInputKeys = join.probeKeys
            .filter(calcFields(_).isInstanceOf[RexInputRef])
            .map(calcFields(_).asInstanceOf[RexInputRef].getIndex)
        rfCalls.foreach { call =>
          call.getOperands()(0) match {
            case ref: RexInputRef =>
              if (probeInputKeys.contains(ref.getIndex)) {
                val rf = call.getOperator.asInstanceOf[SqlRuntimeFilterFunction]
                rf.builder.filters -= rf
                toRemove += rf
              }
            case _ =>
          }
        }

        if (toRemove.nonEmpty) {
          val newCalc = removeFilters(calc, toRemove.toArray)
          call.transformTo(join.copy(join.getTraitSet, join.getInputs.map { input =>
            if (input == join.probeRel) newCalc else input
          }))
        }
      case _ =>
    }
  }
}

object UselessBroadcastRfRemoveRule {
  val INSTANCE = new UselessBroadcastRfRemoveRule
}
