/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.plan.schema

import org.apache.flink.table.plan.stats.FlinkStatistic

import org.apache.calcite.rel.RelNode
import org.apache.calcite.rel.`type`.{RelDataType, RelDataTypeFactory}

/**
  * An intermediate Table to wrap a optimized RelNode inside. The input data of this Table is
  * generated by the underlying optimized RelNode.
  *
  * @param relNode       underlying optimized RelNode
  * @param isAccRetract  true if input data of table contain retraction messages.
  * @param statistic     statistics of current Table
  */
class IntermediateRelNodeTable(
    val relNode: RelNode,
    val isAccRetract: Boolean,
    val statistic: FlinkStatistic = FlinkStatistic.UNKNOWN)
  extends FlinkTable {

  def this(relNode: RelNode) {
    this(relNode, false)
  }

  /**
    * Creates a copy of this table, changing statistic.
    *
    * @param statistic A new FlinkStatistic.
    * @return Copy of this table, substituting statistic.
    */
  override def copy(statistic: FlinkStatistic): FlinkTable =
    new IntermediateRelNodeTable(relNode, isAccRetract, statistic)

  override def getRowType(typeFactory: RelDataTypeFactory): RelDataType = relNode.getRowType

  /**
    * Returns statistics of current table
    *
    * @return statistics of current table
    */
  override def getStatistic: FlinkStatistic = statistic

}
