/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.plan.stats;

import org.apache.flink.util.Preconditions;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Objects;

/**
 * Column statistics.
 * Note we currently assume that, in Flink, the max and min of ColumnStats will be same type as
 * the Flink column type. For example, for SHORT and Long columns, the max and min of their
 * ColumnStats should be of type SHORT and LONG. This assumption might change in the future.
 */
public class ColumnStats implements Serializable {

	private static final long serialVersionUID = -1L;

	public final Long ndv;

	public final Long nullCount;

	public final Double avgLen;

	public final Integer maxLen;

	public final Object max;

	public final Object min;

	/*
	 * Initialize a ColumnStats
	 * @param ndv       number of distinct values (it may be an inaccurate value.)
	 * @param nullCount number of nulls (it may be an inaccurate value.)
	 * @param avgLen    average length of column values (it may be an inaccurate value.)
	 * @param maxLen    max length of column values (it may be an inaccurate value.)
	 * @param max       max value of column values (it must be a correct value,
	 *                  the accurate max value may be less than this value.)
	 * @param min       min value of column values (it must be a correct value,
	 *                  the accurate min value may be greater than this value.)
	 */
	public ColumnStats(Long ndv, Long nullCount, Double avgLen, Integer maxLen, Object max, Object min) {
		Preconditions.checkArgument(max == null || max instanceof Comparable);
		Preconditions.checkArgument(min == null || min instanceof Comparable);
		Preconditions.checkArgument(max == null || min == null || max.getClass() == min.getClass());

		this.ndv = ndv;
		this.nullCount = nullCount;
		this.avgLen = avgLen;
		this.maxLen = maxLen;
		this.max = max;
		this.min = min;
	}

	public ColumnStats(Long ndv) {
		this(ndv, null, null, null, null, null);
	}

	public Long ndv() {
		return ndv;
	}

	public Long nullCount() {
		return nullCount;
	}

	public Double avgLen() {
		return avgLen;
	}

	public Integer maxLen() {
		return maxLen;
	}

	public Object max() {
		return max;
	}

	public Object min() {
		return min;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}

		if (obj == null || this.getClass() != obj.getClass()) {
			return false;
		}

		ColumnStats other = (ColumnStats) obj;
		return (Objects.equals(ndv, other.ndv)
			&& Objects.equals(nullCount, other.nullCount)
			&& Objects.equals(avgLen, other.avgLen)
			&& Objects.equals(maxLen, other.maxLen)
			&& Objects.equals(max, other.max)
			&& Objects.equals(min, other.min));
	}

	@Override
	public String toString() {
		String[] stats = new String[] {
			(ndv != null) ? "ndv=" + ndv : "",
			(nullCount != null) ? "nullCount=" + nullCount : "",
			(avgLen != null) ? "avgLen=" + avgLen : "",
			(maxLen != null) ? "maxLen=" + maxLen : "",
			(max != null) ? "max=" + max : "",
			(min != null) ? "min=" + min : ""
		};
		stats = Arrays.stream(stats).filter(s -> !s.isEmpty()).toArray(String[]::new);
		return String.format("ColumnStats(%s)", String.join(", ", stats));
	}
}
