/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.plan.stats;

import org.apache.flink.util.Preconditions;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * Table statistics.
 */
public class TableStats implements Serializable {

	private static final long serialVersionUID = -1L;

	public Long rowCount;

	public Map<String, ColumnStats> colStats;

	public Map<String, List<Object>> skewInfo;

	public static final TableStats UNKNOWN_STATS = new TableStats(null);

	public TableStats(
		Long rowCount,
		Map<String, ColumnStats> colStats,
		Map<String, List<Object>> skewInfo) {

		Preconditions.checkArgument(rowCount == null || rowCount >= 0L, "rowCount cannot be negative!");
		Preconditions.checkArgument(colStats != null, "column statistic cannot be null!");
		Preconditions.checkArgument(skewInfo != null, "skew information cannot be null!");

		this.rowCount = rowCount;
		this.colStats = colStats;
		this.skewInfo = skewInfo;
	}

	public TableStats(Long rowCount) {
		this(rowCount, new HashMap<>(), new HashMap<>());
	}

	public TableStats(Long rowCount, Map<String, ColumnStats> colStats) {
		this(rowCount, colStats, new HashMap<>());
	}

	public Long rowCount() {
		return rowCount;
	}

	public Map<String, ColumnStats> colStats() {
		return colStats;
	}

	public Map<String, List<Object>> skewInfo() {
		return skewInfo;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}

		if (obj == null || this.getClass() != obj.getClass()) {
			return false;
		}

		TableStats other = (TableStats) obj;
		return (Objects.equals(rowCount, other.rowCount)
			&& Objects.equals(colStats, other.colStats)
			&& Objects.equals(skewInfo, other.skewInfo));
	}

	@Override
	public String toString() {
		String[] stats = new String[] {
			rowCount != null ? "rowCount=" + rowCount : "",
			!colStats.isEmpty() ? "colStats=" + colStats : "",
			!skewInfo.isEmpty() ? "skewInfo=" + skewInfo : ""
		};
		return String.format("TableStats{%s}", String.join(", ", stats));
	}

	// Work around SI-8911 scala.collection.convert.Wrappers$MapWrapper not serializable
	// https://github.com/scala/scala/pull/4167
	private void writeObject(ObjectOutputStream oos) throws IOException {
		oos.writeObject(rowCount);

		oos.writeObject(colStats.size());
		for (Map.Entry<String, ColumnStats> e : colStats.entrySet()) {
			oos.writeObject(e.getKey());
			oos.writeObject(e.getValue());
		}

		oos.writeObject(skewInfo.size());
		for (Map.Entry<String, List<Object>> e : skewInfo.entrySet()) {
			oos.writeObject(e.getKey());
			oos.writeObject(e.getValue());
		}
	}

	private void readObject(ObjectInputStream ois) throws ClassNotFoundException, IOException {
		this.rowCount = (Long) ois.readObject();

		this.colStats = new HashMap<>();
		int size = (int) ois.readObject();
		for (int i = 0; i < size; i++) {
			this.colStats.put((String) ois.readObject(), (ColumnStats) ois.readObject());
		}

		this.skewInfo = new HashMap<>();
		int size2 = (int) ois.readObject();
		for (int i = 0; i < size2; i++) {
			this.skewInfo.put((String) ois.readObject(), (List<Object>) ois.readObject());
		}
	}

	public static Builder builder() {
		return new Builder();
	}

	/**
	 * Builder for creating a {@link TableStats}.
	 */
	public static class Builder {
		private Long rowCount;
		private Map<String, ColumnStats> colStats;
		private Map<String, List<Object>> skewInfo;

		public Builder() {
			rowCount = null;
			colStats = new HashMap<>();
			skewInfo = new HashMap<>();
		}

		public Builder rowCount(Long rowCount) {
			this.rowCount = rowCount;
			return this;
		}

		public Builder colStats(Map<String, ColumnStats> colStats) {
			this.colStats = colStats;
			return this;
		}

		public Builder skewInfo(Map<String, List<Object>> skewInfo) {
			this.skewInfo = skewInfo;
			return this;
		}

		public Builder tableStats(TableStats stats) {
			Preconditions.checkArgument(stats != null, "input TableStats cannot be null!");
			this.rowCount = stats.rowCount;
			this.colStats = stats.colStats;
			this.skewInfo = stats.skewInfo;
			return this;
		}

		public TableStats build() {
			return new TableStats(rowCount, colStats, skewInfo);
		}
	}
}
