/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.plan.util

import org.apache.flink.table.api.window.{CountWindow, TimeWindow}
import org.apache.flink.table.api.{TableConfig, TableConfigOptions, TableException}
import org.apache.flink.table.calcite.FlinkRelBuilder.NamedWindowProperty
import org.apache.flink.table.calcite.FlinkTypeFactory
import org.apache.flink.table.codegen.agg.AggsHandlerCodeGenerator
import org.apache.flink.table.codegen.{CodeGeneratorContext, GeneratedSubKeyedAggsHandleFunction}
import org.apache.flink.table.expressions.ExpressionUtils._
import org.apache.flink.table.expressions.{ProctimeAttribute, RowtimeAttribute, WindowEnd, WindowStart}
import org.apache.flink.table.plan.logical.{LogicalWindow, SessionGroupWindow, SlidingGroupWindow, TumblingGroupWindow}
import org.apache.flink.table.types.{DataTypes, InternalType}

import org.apache.calcite.rel.`type`.RelDataType
import org.apache.calcite.tools.RelBuilder

object WindowAggregateUtil {
  /**
    * Return true if window minibatch can be used. False otherwise.
    */
  def isWindowMiniBatchApplicable(
    config: TableConfig,
    window: LogicalWindow,
    aggInfos: Array[AggregateInfo]): Boolean = {
    if (config.getConf.getBoolean(TableConfigOptions.SQL_EXEC_MINIBATCH_WINDOW_ENABLED) &&
      AggregateUtil.doAllSupportPartialMerge(aggInfos)) {
      window match {
        case TumblingGroupWindow(_, timeField, size)
          if isRowtimeAttribute(timeField) && isTimeIntervalLiteral(size) => true
        case SlidingGroupWindow(_, timeField, size, slide)
          if isRowtimeAttribute(timeField) && isTimeIntervalLiteral(size) =>
          val sizeDuration = toDuration(size)
          val slideDuration = toDuration(slide)
          sizeDuration.toMillis % slideDuration.toMillis == 0
        case _ => false
      }
    } else {
      false
    }
  }

  /**
    * Derives output row type for local window aggregate
    */
  def inferLocalWindowAggType(
    aggInfoList: AggregateInfoList,
    inputType: RelDataType,
    groupSet: Array[Int],
    typeFactory: FlinkTypeFactory): RelDataType = {
    val accTypes = aggInfoList.getAccTypes
    val groupingTypes = groupSet
      .map(inputType.getFieldList.get(_).getType)
      .map(FlinkTypeFactory.toInternalType)
    val groupingNames = groupSet.map(inputType.getFieldNames.get(_))
    val accFieldNames = AggregateUtil.inferAggAccumulatorNames(aggInfoList)

    typeFactory.buildRelDataType(
      groupingNames ++ Array("assignedWindow$") ++ accFieldNames,
      groupingTypes ++ Array(DataTypes.TIMESTAMP) ++ accTypes.map(_.toInternalType))
  }

  /**
    * Computes the positions of (window start, window end, row time).
    */
  private[flink] def computeWindowPropertyPos(
    properties: Seq[NamedWindowProperty]): (Option[Int], Option[Int], Option[Int]) = {

    val propPos = properties.foldRight(
      (None: Option[Int], None: Option[Int], None: Option[Int], 0)) {
      case (p, (s, e, rt, i)) => p match {
        case NamedWindowProperty(_, prop) =>
          prop match {
            case WindowStart(_) if s.isDefined =>
              throw new TableException(
                "Duplicate window start property encountered. This is a bug.")
            case WindowStart(_) =>
              (Some(i), e, rt, i - 1)
            case WindowEnd(_) if e.isDefined =>
              throw new TableException("Duplicate window end property encountered. This is a bug.")
            case WindowEnd(_) =>
              (s, Some(i), rt, i - 1)
            case RowtimeAttribute(_) if rt.isDefined =>
              throw new TableException(
                "Duplicate window rowtime property encountered. This is a bug.")
            case RowtimeAttribute(_) =>
              (s, e, Some(i), i - 1)
            case ProctimeAttribute(_) =>
              // ignore this property, it will be null at the position later
              (s, e, rt, i - 1)
          }
      }
    }
    (propPos._1, propPos._2, propPos._3)
  }

  /**
    * generate window aggsHandler for window aggregator.
    */
  def createAggsHandler(
      name: String,
      window: LogicalWindow,
      namedProperties: Seq[NamedWindowProperty],
      aggInfoList: AggregateInfoList,
      config: TableConfig,
      relBuilder: RelBuilder,
      inputFieldTypes: Seq[InternalType],
      needRetraction: Boolean,
      minibatch: Boolean,
      mergedAccOffset: Int,
      mergedAccOnHeap: Boolean): GeneratedSubKeyedAggsHandleFunction[_] = {

    val ctx = CodeGeneratorContext(config, supportReference = true)
    val mergingWindow = window match {
      case SlidingGroupWindow(_, _, size, _) if isTimeIntervalLiteral(size) => true
      case SessionGroupWindow(_, _, _) => true
      case _ => false
    }
    val windowClass = window match {
      case TumblingGroupWindow(_, _, size) if isRowCountLiteral(size) => classOf[CountWindow]
      case SlidingGroupWindow(_, _, size, _) if isRowCountLiteral(size) => classOf[CountWindow]
      case _ => classOf[TimeWindow]
    }

    val aggsGenerator = new AggsHandlerCodeGenerator(
      ctx,
      relBuilder,
      inputFieldTypes,
      needRetraction,
      mergingWindow || minibatch,
      config.getNullCheck,
      copyInputField = minibatch)

    aggsGenerator.withMerging(
      mergedAccOffset,
      mergedAccOnHeap,
      aggInfoList.getAccTypes)

    aggsGenerator.generateSubKeyedAggsHandler(
      name,
      aggInfoList,
      namedProperties.map(_.property),
      windowClass)
  }
}
