/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.runtime.bundle;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.dropwizard.metrics.DropwizardMeterWrapper;
import org.apache.flink.metrics.Counter;
import org.apache.flink.metrics.Gauge;
import org.apache.flink.metrics.Meter;
import org.apache.flink.runtime.state.VoidNamespace;
import org.apache.flink.runtime.state.VoidNamespaceSerializer;
import org.apache.flink.runtime.state.keyed.KeyedValueState;
import org.apache.flink.runtime.state.keyed.KeyedValueStateDescriptor;
import org.apache.flink.streaming.api.SimpleTimerService;
import org.apache.flink.streaming.api.TimerService;
import org.apache.flink.streaming.api.operators.InternalTimer;
import org.apache.flink.streaming.api.operators.InternalTimerService;
import org.apache.flink.streaming.api.operators.OneInputStreamOperator;
import org.apache.flink.streaming.api.watermark.Watermark;
import org.apache.flink.streaming.runtime.streamrecord.StreamRecord;
import org.apache.flink.table.dataformat.BaseRow;
import org.apache.flink.table.dataformat.util.BaseRowUtil;
import org.apache.flink.table.runtime.fault.tolerant.TriggerableOperator;
import org.apache.flink.table.runtime.util.StreamRecordCollector;
import org.apache.flink.table.typeutils.BaseRowTypeInfo;
import org.apache.flink.util.Collector;
import org.apache.flink.util.Preconditions;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * This stream operator is used for get first/lastRow on rowtime of every key partition
 * in rowtime mode. The emitting of First/LastRow result is triggered by watermark, and
 * buffering operations of MiniBatch are completed during the watermark interval.
 * Note: LastRow mode is not supported now.
 */
public class RowtimeDeduplicateOperator
	extends TriggerableOperator<BaseRow, VoidNamespace, BaseRow>
	implements OneInputStreamOperator<BaseRow, BaseRow> {

	private static final long serialVersionUID = 1L;

	private static final String LATE_ELEMENTS_DROPPED_METRIC_NAME = "numLateRecordsDropped";
	private static final String LATE_ELEMENTS_DROPPED_RATE_METRIC_NAME = "lateRecordsDroppedRate";
	private static final String WATERMARK_LATENCY_METRIC_NAME = "watermarkLatency";

	private final int rowtimeIndex;

	private final BaseRowTypeInfo valueType;

	private transient Collector<BaseRow> collector;

	private transient Map<BaseRow, BaseRow> buffer;

	private transient TimerService timerService;

	private transient KeyedValueState<BaseRow, BaseRow> pkRow;

	private transient TypeSerializer<BaseRow> valueSer;

	private transient Counter numLateRecordsDropped;

	private transient Meter lateRecordsDroppedRate;

	private transient Gauge<Long> watermarkLatency;

	private final long minibatchSize;

	private transient int numOfElements;

	public RowtimeDeduplicateOperator(
		int rowtimeIndex,
		BaseRowTypeInfo valueType,
		long minibatchSize) {
		Preconditions.checkArgument(rowtimeIndex >= 0);

		this.rowtimeIndex = rowtimeIndex;
		this.valueType = valueType;
		this.minibatchSize = minibatchSize;
	}

	@Override
	public void open() throws Exception {
		super.open();
		this.buffer = new HashMap<>();
		this.collector = new StreamRecordCollector<>(output);

		this.numOfElements = 0;
		// create & restore state
		this.valueSer = valueType.createSerializer();

		//noinspection unchecked
		KeyedValueStateDescriptor<BaseRow, BaseRow> rowStateDesc = new KeyedValueStateDescriptor<>(
			"rowState", (TypeSerializer) getKeySerializer(), valueSer);
		this.pkRow = getKeyedState(rowStateDesc);

		InternalTimerService<VoidNamespace> internalTimerService =
			getInternalTimerService("deduplicate-timers", VoidNamespaceSerializer.INSTANCE, this);
		this.timerService = new SimpleTimerService(internalTimerService);

		// metrics
		this.numLateRecordsDropped = metrics.counter(LATE_ELEMENTS_DROPPED_METRIC_NAME);
		this.lateRecordsDroppedRate = metrics.meter(
			LATE_ELEMENTS_DROPPED_RATE_METRIC_NAME,
			new DropwizardMeterWrapper(new com.codahale.metrics.Meter()));
		this.watermarkLatency = metrics.gauge(WATERMARK_LATENCY_METRIC_NAME, () -> {
			long watermark = this.timerService.currentWatermark();
			if (watermark < 0) {
				return 0L;
			} else {
				return System.currentTimeMillis() - watermark;
			}
		});
	}

	@Override
	public void processElement(StreamRecord<BaseRow> element) throws Exception {
		numOfElements++;
		BaseRow inputRow = element.getValue();
		long timestamp = inputRow.getLong(rowtimeIndex);
		// ignore late records.
		if (timestamp <= timerService.currentWatermark()) {
			numLateRecordsDropped.inc();
			lateRecordsDroppedRate.markEvent();
			return;
		}

		// should be accumulate msg.
		Preconditions.checkArgument(BaseRowUtil.isAccumulateMsg(inputRow));

		BaseRow key = (BaseRow) getCurrentKey();
		BaseRow value = buffer.get(key);
		// update entry value of the specific key in the buffer.
		if (value == null || isFirstRow(value, inputRow)) {
			buffer.put(key, valueSer.copy(inputRow));
			if (minibatchSize > 0 && numOfElements > minibatchSize) {
				finishBundle();
			}
		}
	}

	/**
	 * flush buffer to state
	 * FirstRow mode: emitting of results is triggered by watermark, no retraction.
	 * LastRow mode is not supported now.
	 */
	private void finishBundle() {
		this.numOfElements = 0;

		// flush buffer to state.
		Map<BaseRow, BaseRow> pkRowsMap = pkRow.getAll(buffer.keySet());
		Iterator<Map.Entry<BaseRow, BaseRow>> iter = buffer.entrySet().iterator();
		while (iter.hasNext()) {
			Map.Entry<BaseRow, BaseRow> entry = iter.next();
			BaseRow currentKey = entry.getKey();
			BaseRow currentRow = entry.getValue();
			BaseRow prevRow = pkRowsMap.get(currentKey);

			deduplicateKeepFirstRow(prevRow, currentKey, currentRow);
		}
		buffer.clear();
	}

	@Override
	public void prepareSnapshotPreBarrier(long checkpointId) throws Exception {
		finishBundle();
	}

	@Override
	public void endInput() throws Exception {
	}

	protected Counter getNumLateRecordsDropped() {
		return numLateRecordsDropped;
	}

	protected Gauge<Long> getWatermarkLatency() {
		return watermarkLatency;
	}

	@Override
	public boolean requireState() {
		return true;
	}

	// ------------------------------------------------------------------------------
	// process functions.
	// ------------------------------------------------------------------------------
	private void deduplicateKeepFirstRow(
		BaseRow prevRow,
		BaseRow currentKey,
		BaseRow currentRow) {
		if (!isFirstRow(prevRow, currentRow)) {
			return;
		}

		pkRow.put(currentKey, currentRow);
		// update timer for a specific key.
		setCurrentKey(currentKey);
		if (prevRow != null) {
			timerService.deleteEventTimeTimer(prevRow.getLong(rowtimeIndex));
		}
		timerService.registerEventTimeTimer(currentRow.getLong(rowtimeIndex));
	}

	private boolean isFirstRow(BaseRow prevRow, BaseRow currentRow) {
		return prevRow == null || currentRow.getLong(rowtimeIndex) < prevRow.getLong(rowtimeIndex);
	}

	@Override
	public void processWatermark(Watermark mark) throws Exception {
		// flush buffer before advance watermark.
		finishBundle();
		super.processWatermark(mark);
	}

	@Override
	public void onEventTime(InternalTimer<BaseRow, VoidNamespace> timer) throws Exception {
		collector.collect(pkRow.get(timer.getKey()));
	}

	@Override
	public void onProcessingTime(InternalTimer<BaseRow, VoidNamespace> timer) throws Exception {
		// do nothing.
	}
}
