/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.runtime.fault.tolerant;

import org.apache.flink.annotation.Internal;
import org.apache.flink.util.Collector;

/**
 * A Collector proxy which wraps the real collector. The proxy catches all exception of the underlying collector when
 * emits a record, then wraps the exception in a {@link NonnegligibleException}.
 *
 * @param <T> The type of the elements that can be emitted.
 */
@Internal
public class CollectorProxy<T> implements Collector<T> {

	protected final Collector<T> collector;

	public CollectorProxy(Collector<T> collector) {
		this.collector = collector;
	}

	@Override
	public void collect(T record) {
		try {
			collector.collect(record);
		} catch (Throwable e) {
			throw new NonnegligibleException(e);
		}
	}

	@Override
	public void close() {
		this.collector.close();
	}
}
