/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.runtime.fault.tolerant;

import org.apache.flink.metrics.Counter;
import org.apache.flink.runtime.jobgraph.JobVertexID;
import org.apache.flink.streaming.api.datastream.AsyncDataStream;
import org.apache.flink.streaming.api.functions.async.AsyncFunction;
import org.apache.flink.streaming.api.graph.StreamConfig;
import org.apache.flink.streaming.api.operators.Output;
import org.apache.flink.streaming.runtime.streamrecord.StreamRecord;
import org.apache.flink.streaming.runtime.tasks.StreamTask;

/**
 * An subclass of {@link org.apache.flink.streaming.api.operators.async.AsyncWaitOperator} which is fault tolerant.
 *
 * @param <IN> Input type for the operator.
 * @param <OUT> Output type for the operator.
 */
public class FaultTolerantableAsyncWaitOperator<IN, OUT> extends org.apache.flink.streaming.api.operators.async.AsyncWaitOperator<IN, OUT>
		implements FaultTolerantableOperator<OUT> {

	/** The name of underlying operator. */
	private final String operatorName;

	private boolean faultTolerantEnable;

	private Counter ignoreCnt;

	/** The task that contains this operator (and other operators in the same chain). */
	private JobVertexID jobVertexId;

	public FaultTolerantableAsyncWaitOperator(
			AsyncFunction<IN, OUT> asyncFunction,
			long timeout,
			int capacity,
			AsyncDataStream.OutputMode outputMode,
			String operatorName) {
		super(asyncFunction, timeout, capacity, outputMode);
		this.operatorName = operatorName;
	}

	@Override
	public void setup(StreamTask<?, ?> containingTask, StreamConfig config, Output<StreamRecord<OUT>> output) {
		if (faultTolerantEnable) {
			super.setup(containingTask, config, new OutputProxy<>(output));
			jobVertexId = containingTask.getEnvironment().getJobVertexId();
		} else {
			super.setup(containingTask, config, output);
		}
	}

	@Override
	public void open() throws Exception {
		super.open();
		if (faultTolerantEnable) {
			ignoreCnt = getMetricGroup().counter(FaultTolerantUtil.IGNORE_CNT_METRIC_NAME);
		}
	}

	@Override
	public void failOperator(Throwable throwable) {
		if (faultTolerantEnable) {
			if (throwable instanceof Exception) {
				if (throwable instanceof NonnegligibleException) {
					super.failOperator(throwable);
				} else {
					LOG.error("An exception is ignored in operator [{}] of vertex [{}].\n" +
									"The output maybe discarded.",
							operatorName, jobVertexId, throwable);
					ignoreCnt.inc();
				}
			} else {
				super.failOperator(throwable);
			}
		} else {
			super.failOperator(throwable);
		}
	}

	@Override
	public void enableFaultTolerant() {
		this.faultTolerantEnable = true;
	}
}
