/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.runtime.fault.tolerant;

import org.apache.flink.metrics.Counter;
import org.apache.flink.streaming.api.operators.AbstractStreamOperator;
import org.apache.flink.streaming.runtime.tasks.ProcessingTimeCallback;
import org.apache.flink.streaming.runtime.tasks.ProcessingTimeService;

import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

/**
 * A {@link ProcessingTimeCallback} operator which catches all exception happen in {@link ProcessingTimeCallback#onProcessingTime}.
 * Note:
 * The exception happens in other cases (e.g processElement) are handled by {@link AbstractStreamOperatorProxy} instead of
 * {@link ProcessingTimeCallbackOperator}.
 *
 * @param <OUT> The output type of the operator
 */
public abstract class ProcessingTimeCallbackOperator<OUT> extends AbstractStreamOperator<OUT>
		implements ProcessingTimeCallback, FaultTolerantableOperator<OUT> {

	private transient Counter ignoreCnt;
	private boolean faultTolerantEnable = false;

	@Override
	public void open() throws Exception {
		super.open();
		if (faultTolerantEnable) {
			ignoreCnt = getMetricGroup().counter(FaultTolerantUtil.IGNORE_CNT_METRIC_NAME);
		}
	}

	@Override
	public void enableFaultTolerant() {
		this.faultTolerantEnable = true;
	}

	@Override
	protected ProcessingTimeService getProcessingTimeService() {
		ProcessingTimeService processingTimeService = super.getProcessingTimeService();
		if (!faultTolerantEnable) {
			return processingTimeService;
		}
		return new ProcessingTimeService() {

			@Override
			public long getCurrentProcessingTime() {
				return processingTimeService.getCurrentProcessingTime();
			}

			@Override
			public ScheduledFuture<?> registerTimer(long timestamp, ProcessingTimeCallback target) {
				ProcessingTimeCallback t = timestamp1 -> {
					try {
						target.onProcessingTime(timestamp1);
					} catch (Exception e) {
						// Exclude runtime exception
						if (e instanceof NonnegligibleException) {
							throw e;
						} else {
							LOG.error("An exception is ignored trigger on ProcessingTime.\n" +
											"The output maybe discarded.\n" +
											"Timestamp : [{}].",
									timestamp1, e);
							ignoreCnt.inc();
						}
					}
				};

				return processingTimeService.registerTimer(timestamp, t);
			}

			@Override
			public ScheduledFuture<?> scheduleAtFixedRate(ProcessingTimeCallback callback, long initialDelay,
					long period) {
				return processingTimeService.scheduleAtFixedRate(callback, initialDelay, period);
			}

			@Override
			public boolean isTerminated() {
				return processingTimeService.isTerminated();
			}

			@Override
			public void quiesce() throws InterruptedException {
				processingTimeService.quiesce();
			}

			@Override
			public void awaitPendingAfterQuiesce() throws InterruptedException {
				processingTimeService.awaitPendingAfterQuiesce();
			}

			@Override
			public void shutdownService() {
				processingTimeService.shutdownService();
			}

			@Override
			public boolean shutdownServiceUninterruptible(long timeoutMs) {
				return processingTimeService.shutdownServiceUninterruptible(timeoutMs);
			}

			@Override
			public boolean shutdownAndAwaitPending(long time, TimeUnit timeUnit) throws InterruptedException {
				return processingTimeService.shutdownAndAwaitPending(time, timeUnit);
			}
		};
	}

}
