/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.runtime.fault.tolerant;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.metrics.Counter;
import org.apache.flink.streaming.api.operators.AbstractStreamOperator;
import org.apache.flink.streaming.api.operators.InternalTimer;
import org.apache.flink.streaming.api.operators.InternalTimerService;
import org.apache.flink.streaming.api.operators.Triggerable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serializable;

/**
 * A {@link Triggerable} operator which catches all exception happen in {@link Triggerable#onEventTime} and {@link Triggerable#onProcessingTime}.
 * The exception happens in other cases (e.g processElement) are handled by {@link AbstractStreamOperatorProxy} instead of
 * {@link TriggerableOperator}.
 *
 * @param <K> Type of the keys to which timers are scoped
 * @param <N> Type of the namespace to which timers are scoped
 * @param <OUT> The output type of the operator
 */
public abstract class TriggerableOperator<K, N, OUT> extends AbstractStreamOperator<OUT> implements Triggerable<K, N>,
		FaultTolerantableOperator<OUT> {

	private transient Counter ignoreCnt;
	private boolean faultTolerantEnable = false;

	@Override
	public void open() throws Exception {
		super.open();
		if (faultTolerantEnable) {
			ignoreCnt = getMetricGroup().counter(FaultTolerantUtil.IGNORE_CNT_METRIC_NAME);
		}
	}

	@Override
	public void enableFaultTolerant() {
		this.faultTolerantEnable = true;
	}

	@Override
	public <K, N> InternalTimerService<N> getInternalTimerService(String name, TypeSerializer<N> namespaceSerializer,
			Triggerable<K, N> triggerable) {
		Triggerable<K, N> t;
		if (faultTolerantEnable) {
			t = new TriggerableProxy(ignoreCnt, triggerable);
		} else {
			t = triggerable;
		}
		return super.getInternalTimerService(name, namespaceSerializer, t);
	}


	/**
	 * A proxy wraps the a {@link Triggerable} operator to provide fault-tolerant feature.
	 *
	 * @param <K> Type of the keys to which timers are scoped.
	 * @param <N> Type of the namespace to which timers are scoped.
	 */
	static class TriggerableProxy<K, N> implements Triggerable<K, N>, Serializable {

		private static final long serialVersionUID = 1L;

		private static final Logger LOG = LoggerFactory.getLogger(TriggerableProxy.class);

		private final Counter ignoreCnt;

		private final Triggerable<K, N> triggerable;

		public TriggerableProxy(Counter ignoreCnt, Triggerable<K, N> triggerable) {
			this.ignoreCnt = ignoreCnt;
			this.triggerable = triggerable;
		}

		@Override
		public void onEventTime(InternalTimer<K, N> timer) throws Exception {
			try {
				triggerable.onEventTime(timer);
			} catch (Exception e) {
				// Exclude runtime exception
				if (e instanceof NonnegligibleException) {
					throw e;
				} else {
					LOG.error("An exception is ignored trigger on EventTime.\n" +
									"The output maybe discarded.\n" +
									"Detail information about timer is: \n" +
									"timestamp : [{}], \n" +
									"key : [{}], \n" +
									"namespace : [{}].",
							timer.getTimestamp(), timer.getKey(), timer.getNamespace(), e);
					ignoreCnt.inc();
				}
			}
		}

		@Override
		public void onProcessingTime(InternalTimer<K, N> timer) throws Exception {
			try {
				triggerable.onProcessingTime(timer);
			} catch (Exception e) {
				// Exclude runtime exception
				if (e instanceof NonnegligibleException) {
					throw e;
				} else {
					LOG.error("An exception is ignored trigger on ProcessingTime.\n" +
									"The output maybe discarded.\n" +
									"Detail information about timer is: \n" +
									"timestamp : [{}], \n" +
									"key : [{}], \n" +
									"namespace : [{}].",
							timer.getTimestamp(), timer.getKey(), timer.getNamespace(), e);
					ignoreCnt.inc();
				}
			}
		}
	}

}
