/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.runtime.fault.tolerant;

import org.apache.flink.annotation.Internal;
import org.apache.flink.streaming.api.operators.TwoInputSelection;
import org.apache.flink.streaming.api.operators.TwoInputStreamOperator;
import org.apache.flink.streaming.api.watermark.Watermark;
import org.apache.flink.streaming.runtime.streamrecord.LatencyMarker;
import org.apache.flink.streaming.runtime.streamrecord.StreamRecord;

import java.io.Serializable;

/**
 * A stream operator proxy which wraps the a {@link TwoInputStreamOperator}, catch all exception (except {@link NonnegligibleException})
 * of the underlying operator to prevent job failover.
 *
 * @param <IN1> The input type of the operator
 * @param <IN2> The input type of the operator
 * @param <OUT> The output type of the operator
 */
@Internal
public final class TwoInputOperatorProxy<IN1, IN2, OUT> extends AbstractStreamOperatorProxy<OUT>
		implements TwoInputStreamOperator<IN1, IN2, OUT>, Serializable {

	private static final long serialVersionUID = 1L;

	public TwoInputOperatorProxy(String operatorName, TwoInputStreamOperator<IN1, IN2, OUT> operator) {
		super(operatorName, operator);
	}

	@Override
	public TwoInputSelection processElement1(StreamRecord<IN1> element) throws Exception {
		try {
			return ((TwoInputStreamOperator) operator).processElement1(element);
		} catch (Exception e) {
			// Exclude runtime exception
			if (e instanceof NonnegligibleException) {
				throw e;
			} else {
				LOG.error("An exception is ignored when process element1 in operator [{}] of vertex [{}].\n" +
								"The output for input record [{}] maybe discard.",
						operatorName, jobVertexId, element, e);
				ignoreCnt.inc();
				// Notes: Return TwoInputSelection.ANY is not right for HashJoinOperator in Batch mode.
				return TwoInputSelection.ANY;
			}
		}
	}

	@Override
	public TwoInputSelection processElement2(StreamRecord<IN2> element) throws Exception {
		try {
			return ((TwoInputStreamOperator) operator).processElement2(element);
		} catch (Exception e) {
			// Exclude runtime exception
			if (e instanceof NonnegligibleException) {
				throw e;
			} else {
				LOG.error("An exception is ignored when process element2 in operator [{}] of vertex [{}].\n" +
								"The output for input record [{}] maybe discard.",
						operatorName, jobVertexId, element, e);
				ignoreCnt.inc();
				// Notes: Return TwoInputSelection.ANY is not right for HashJoinOperator in Batch mode.
				return TwoInputSelection.ANY;
			}
		}
	}

	@Override
	public TwoInputSelection firstInputSelection() {
		return ((TwoInputStreamOperator) operator).firstInputSelection();
	}

	@Override
	public void processWatermark1(Watermark mark) throws Exception {
		try {
			((TwoInputStreamOperator) operator).processWatermark1(mark);
		} catch (Exception e) {
			// Exclude runtime exception
			if (e instanceof NonnegligibleException) {
				throw e;
			} else {
				LOG.error("An exception is ignored when process watermark1 in operator [{}] of vertex [{}].\n" +
								"The output maybe discarded.",
						operatorName, jobVertexId, e);
				ignoreCnt.inc();
			}
		}
	}

	@Override
	public void processWatermark2(Watermark mark) throws Exception {
		try {
			((TwoInputStreamOperator) operator).processWatermark2(mark);
		} catch (Exception e) {
			// Exclude runtime exception
			if (e instanceof NonnegligibleException) {
				throw e;
			} else {
				LOG.error("An exception is ignored when process watermark2 in operator [{}] of vertex [{}].\n" +
								"The output maybe discarded.",
						operatorName, jobVertexId, e);
				ignoreCnt.inc();
			}
		}
	}

	@Override
	public void processLatencyMarker1(LatencyMarker latencyMarker) throws Exception {
		((TwoInputStreamOperator) operator).processLatencyMarker1(latencyMarker);
	}

	@Override
	public void processLatencyMarker2(LatencyMarker latencyMarker) throws Exception {
		((TwoInputStreamOperator) operator).processLatencyMarker2(latencyMarker);
	}

	@Override
	public void endInput1() throws Exception {
		try {
			((TwoInputStreamOperator) operator).endInput1();
		} catch (Exception e) {
			// Exclude runtime exception
			if (e instanceof NonnegligibleException) {
				throw e;
			} else {
				LOG.error("An exception is ignored in operator [{}] of vertex [{}].\n" +
								"The output maybe discarded.",
						operatorName, jobVertexId, e);
				ignoreCnt.inc();
			}
		}
	}

	@Override
	public void endInput2() throws Exception {
		try {
			((TwoInputStreamOperator) operator).endInput2();
		} catch (Exception e) {
			// Exclude runtime exception
			if (e instanceof NonnegligibleException) {
				throw e;
			} else {
				LOG.error("An exception is ignored in operator [{}] of vertex [{}].\n" +
								"The output maybe discarded.",
						operatorName, jobVertexId, e);
				ignoreCnt.inc();
			}
		}
	}

}
