/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.runtime.functions.aggfunctions;

import org.apache.flink.table.api.dataview.MapView;
import org.apache.flink.table.functions.AggregateFunction;
import org.apache.flink.table.runtime.functions.aggfunctions.hyperloglog.HyperLogLogSketchHelper;
import org.apache.flink.table.types.DataType;
import org.apache.flink.table.types.DataTypes;
import org.apache.flink.table.types.DecimalType;
import org.apache.flink.table.types.TypeInfoWrappedDataType;
import org.apache.flink.table.typeutils.BinaryStringTypeInfo;

import static org.apache.flink.table.runtime.functions.aggfunctions.hyperloglog.HyperLogLogSketchHelper.LG_CONFIG_K;


/**
 * Built-in approximate count distinct aggregate function for Stream.
 */
public class StreamApproximateCountDistinct {

	/**
	 * Base class for Hll based approximate count distinct aggregate function.
	 */
	@SuppressWarnings("unchecked")
	public abstract static class ApproximateCountDistinctAggFunction extends AggregateFunction<Long, HllAcc> {

		private static final long serialVersionUID = 264794740037091251L;

		public abstract DataType getValueTypeInfo();

		@Override
		public HllAcc createAccumulator() {
			HllAcc acc = new HllAcc();
			acc.oooFlag = false;
			acc.zeroSlotNum = 1 << LG_CONFIG_K;
			acc.kxq0 = 1 << LG_CONFIG_K;
			acc.kxq1 = 0;
			acc.hipAccum = 0;
			acc.sketch = new MapView(DataTypes.SHORT, DataTypes.BYTE_ARRAY);
			return acc;
		}

		public void accumulate(HllAcc acc, Object input) throws Exception {
			if (input == null) {
				return;
			}
			HyperLogLogSketchHelper.insert(acc, input);
		}

		public void merge(HllAcc acc, Iterable<HllAcc> itr) throws Exception {
			for (HllAcc sk: itr) {
				HyperLogLogSketchHelper.merge(acc, sk);
			}
		}

		@Override
		public Long getValue(HllAcc accumulator) {
			return HyperLogLogSketchHelper.getEstimate(accumulator);
		}

		@Override
		public DataType[] getUserDefinedInputTypes(Class[] signature) {
			if (signature.length == 1) {
				return new DataType[]{getValueTypeInfo()};
			} else if (signature.length == 0) {
				return new DataType[0];
			} else {
				throw new UnsupportedOperationException();
			}
		}
	}

	/**
	 * Accumulator for HyperLogLog, variables name here come from referred paper.
	 */
	public static class HllAcc {
		// indicating if Hip estimator can be used.
		public boolean oooFlag;
		// int zeroSlotNum: number of empty slot in Hll buffer.
		public int zeroSlotNum;
		// incremental sum variable for final Hll estimation.
		public double kxq0;
		// incremental sum variable for final Hll estimation.
		public double kxq1;
		// incremental variable for Hip estimator.
		public double hipAccum;
		// Hll slots buffer.
		public MapView<Short, byte[]> sketch;
	}

	/**
	 * Built-in byte approximate count distinct aggregate function.
	 */
	public static class ByteApproximateCountDistinctAggFunction extends ApproximateCountDistinctAggFunction {
		@Override
		public DataType getValueTypeInfo() {
			return DataTypes.BYTE;
		}
	}

	/**
	 * Built-in decimal approximate count distinct aggregate function.
	 */
	public static class DecimalApproximateCountDistinctAggFunction extends ApproximateCountDistinctAggFunction {

		public final DecimalType decimalType;

		public DecimalApproximateCountDistinctAggFunction(DecimalType decimalType) {
			this.decimalType = decimalType;
		}

		@Override
		public DataType getValueTypeInfo() {
			return this.decimalType;
		}
	}

	/**
	 * Built-in double approximate count distinct aggregate function.
	 */
	public static class DoubleApproximateCountDistinctAggFunction extends ApproximateCountDistinctAggFunction {
		@Override
		public DataType getValueTypeInfo() {
			return DataTypes.DOUBLE;
		}
	}

	/**
	 * Built-in float approximate count distinct aggregate function.
	 */
	public static class FloatApproximateCountDistinctAggFunction extends ApproximateCountDistinctAggFunction {
		@Override
		public DataType getValueTypeInfo() {
			return DataTypes.FLOAT;
		}
	}

	/**
	 * Built-in int approximate count distinct aggregate function.
	 */
	public static class IntApproximateCountDistinctAggFunction extends ApproximateCountDistinctAggFunction {
		@Override
		public DataType getValueTypeInfo() {
			return DataTypes.INT;
		}
	}

	/**
	 * Built-in long approximate count distinct aggregate function.
	 */
	public static class LongApproximateCountDistinctAggFunction extends ApproximateCountDistinctAggFunction {
		@Override
		public DataType getValueTypeInfo() {
			return DataTypes.LONG;
		}
	}

	/**
	 * Built-in short approximate count distinct aggregate function.
	 */
	public static class ShortApproximateCountDistinctAggFunction extends ApproximateCountDistinctAggFunction {
		@Override
		public DataType getValueTypeInfo() {
			return DataTypes.SHORT;
		}
	}

	/**
	 * Built-in boolean approximate count distinct aggregate function.
	 */
	public static class BooleanApproximateCountDistinctAggFunction extends ApproximateCountDistinctAggFunction {
		@Override
		public DataType getValueTypeInfo() {
			return DataTypes.BOOLEAN;
		}
	}

	/**
	 * Built-in date approximate count distinct aggregate function.
	 */
	public static class DateApproximateCountDistinctAggFunction extends ApproximateCountDistinctAggFunction {
		@Override
		public DataType getValueTypeInfo() {
			return DataTypes.DATE;
		}
	}

	/**
	 * Built-in time approximate count distinct aggregate function.
	 */
	public static class TimeApproximateCountDistinctAggFunction extends ApproximateCountDistinctAggFunction {
		@Override
		public DataType getValueTypeInfo() {
			return DataTypes.TIME;
		}
	}

	/**
	 * Built-in timestamp approximate count distinct aggregate function.
	 */
	public static class TimestampApproximateCountDistinctAggFunction extends ApproximateCountDistinctAggFunction {
		@Override
		public DataType getValueTypeInfo() {
			return DataTypes.TIMESTAMP;
		}
	}

	/**
	 * Built-in string approximate count distinct aggregate function.
	 */
	public static class StringApproximateCountDistinctAggFunction extends ApproximateCountDistinctAggFunction {
		@Override
		public DataType getValueTypeInfo() {
			return new TypeInfoWrappedDataType(BinaryStringTypeInfo.INSTANCE);
		}
	}
}
