/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.runtime.window;

import org.apache.flink.dropwizard.metrics.DropwizardMeterWrapper;
import org.apache.flink.metrics.Counter;
import org.apache.flink.metrics.Meter;
import org.apache.flink.runtime.state.StateSnapshotContext;
import org.apache.flink.streaming.runtime.streamrecord.StreamRecord;
import org.apache.flink.table.api.window.TimeWindow;
import org.apache.flink.table.dataformat.BaseRow;
import org.apache.flink.table.runtime.window.aligned.AlignedWindowAggregator;
import org.apache.flink.table.runtime.window.aligned.AlignedWindowTrigger;
import org.apache.flink.table.runtime.window.assigners.WindowAssigner;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * An operator that used both in normal window aggregate and global window aggregate
 * of two stage window optimization.
 * The difference between this operator and {@link LocalAlignedWindowOperator} is that
 * the result emitting of a window is triggered when current watermark passes the
 * window end on this operator, while watermark is treated as an symbol for
 * {@link LocalAlignedWindowOperator} to flush all window buffer accumulators to downstream,
 * no matter whether current watermark passes window end.
 */
public class KeyedAlignedWindowOperator extends AbstractAlignedWindowOperator {

	private static final long serialVersionUID = 1L;

	private transient Counter numLateRecordsDropped;
	private transient Meter lateRecordsDroppedRate;

	public KeyedAlignedWindowOperator(
		AlignedWindowAggregator<BaseRow, TimeWindow, BaseRow> windowRunner,
		WindowAssigner<TimeWindow> windowAssigner,
		AlignedWindowTrigger windowTrigger,
		int rowtimeIndex) {
		super(windowRunner, windowAssigner, windowTrigger, rowtimeIndex);
	}

	@Override
	public void open() throws Exception {
		super.open();
		this.numLateRecordsDropped = metrics.counter(LATE_ELEMENTS_DROPPED_METRIC_NAME);
		this.lateRecordsDroppedRate = metrics.meter(
			LATE_ELEMENTS_DROPPED_RATE_METRIC_NAME,
			new DropwizardMeterWrapper(new com.codahale.metrics.Meter()));
	}

	@Override
	public void processElement(StreamRecord<BaseRow> record) throws Exception {
		// prepare inputRow and timestamp
		BaseRow inputRow = record.getValue();
		// TODO: support processing time in the future
		long timestamp = inputRow.getLong(this.rowtimeIndex);
		// The received record is paned acc in case of global HOP window agg, and
		// it is necessary to assign actual windows that the pane belongs.
		Collection<TimeWindow> windows = windowAssigner.assignWindows(inputRow, timestamp);

		boolean isElementDropped = true;
		for (TimeWindow window : windows) {
			if (!isWindowLate(window, getCurrentWatermark())) {
				isElementDropped = false;
				windowRunner.addElement(getKey(inputRow), window, inputRow);
			}
			// TODO: support late arrival trigger
		}

		if (isElementDropped) {
			numLateRecordsDropped.inc();
			lateRecordsDroppedRate.markEvent();
		}
	}

	@Override
	protected BaseRow getKey(BaseRow input) throws Exception {
		return (BaseRow) getCurrentKey();
	}

	@Override
	public void snapshotState(StateSnapshotContext context) throws Exception {
		super.snapshotState(context);
		windowRunner.snapshot();
	}

	@Override
	protected void advanceWatermark(long watermark) throws Exception {
		if (nextTriggerTime == Long.MIN_VALUE) {
			TimeWindow lowestWindow = windowRunner.lowestWindow();
			if (lowestWindow == null) {
				// if there is no window registered, no trigger happens
				return;
			} else {
				// initial first next trigger time
				this.nextTriggerTime = lowestWindow.maxTimestamp();
				this.nextTriggerWindow = lowestWindow;
			}
			// the initial trigger time should not be too larger than current watermark to
			// avoid skipping some un-arrived windows
			if (watermark < nextTriggerTime) {
				// the next trigger time should be the closest to the watermark
				this.nextTriggerTime = windowTrigger.nextTriggerTime(watermark);
				this.nextTriggerWindow = windowTrigger.nextTriggerWindow(watermark);
			}
		}

		List<TimeWindow> windowsToExpire = new ArrayList<>();
		for (TimeWindow window: windowRunner.ascendingWindows()) {
			if (isWindowLate(window, watermark)) {
				windowsToExpire.add(window);
			}
			if (watermark < nextTriggerTime || window.compareTo(nextTriggerWindow) < 0) {
				continue;
			}
			if (needTriggerWindow(window, watermark)) {
				windowRunner.fireWindow(window);
			} else {
				// the next trigger time should be the closest to the watermark
				this.nextTriggerTime = windowTrigger.nextTriggerTime(watermark);
				this.nextTriggerWindow = windowTrigger.nextTriggerWindow(watermark);
				// if a window can not be triggered, windows following in the iterator
				// will definitely not be late.
				break;
			}
		}
		expireWindows(windowsToExpire);
	}

	@Override
	public boolean requireState() {
		// always requireState
		return true;
	}

	// ------------------------------------------------------------------------------
	// Visible For Testing
	// ------------------------------------------------------------------------------

	protected Counter getNumLateRecordsDropped() {
		return numLateRecordsDropped;
	}
}
