/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.sinks.filesystem

import org.apache.flink.table.codegen.GeneratedProjection
import org.apache.flink.table.dataformat.BaseRow
import org.apache.flink.table.types.RowType

/**
  * OutputFormat writes data using all static partition writes, that means this format only write to
  * one directory (partition).
  *
  * @param description Description about how to output records.
  */
class StaticPartitionOutput(
    path: String,
    description: OutputJobDescription,
    generatedProjectValues: GeneratedProjection,
    partitionSchema: RowType)
  extends DynamicPartitionOutput(
    path,
    description,
    generatedProjectValues,
    partitionSchema) {

  /** Cache for compute partition path. **/
  private var staticPartitionPath: Option[String] = Option.empty

  /** Flag saying whether the OutputFormat is refreshed, only used when all partitions
    * are static. */
  private var formatRefreshed: Boolean = false

  override def getPartitionPath(partitionKey: Option[BaseRow]): Option[String] = {
    if (staticPartitionPath.nonEmpty) {
      staticPartitionPath
    } else {
      staticPartitionPath = super.getPartitionPath(partitionKey)
      staticPartitionPath
    }
  }

  override def writeRecord(record: BaseRow): Unit = {
    if (!formatRefreshed) {
      refreshCurrentFormat(Some(getPartitionValues(record)))
      formatRefreshed = true
    }
    currentFormat.writeRecord(record)
  }
}
