/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.sinks.filesystem.parquet

import org.apache.flink.api.common.io.OutputFormat
import org.apache.flink.core.fs.FileSystem.WriteMode
import org.apache.flink.core.fs.Path
import org.apache.flink.table.dataformat.BaseRow
import org.apache.flink.table.factories.parquet.ParquetOptions
import org.apache.flink.table.sinks.filesystem.{FileSystemOptions, OutputFormatFactory}
import org.apache.flink.table.sinks.parquet.RowParquetOutputFormat
import org.apache.flink.table.types.{AtomicType, DataType, RowType, TypeInfoWrappedDataType}
import org.apache.flink.table.util.TableProperties

import org.apache.hadoop.fs.FileUtil
import org.apache.parquet.hadoop.metadata.CompressionCodecName

import java.io.File

/** [[OutputFormatFactory]] to create a [[OutputFormat]] of parquet format. */
class ParquetFormatFactory extends OutputFormatFactory {
  private[this] var options: java.util.Map[String, String] = _

  override def configure(options: java.util.Map[String, String]): Unit = {
    this.options = options
  }

  override def getFileExtension(taskId: Int): String = {
    ".parquet"
  }

  override def supportDataSchema(dataType: DataType): Boolean = dataType match {
    case _: AtomicType => true
    case udt: TypeInfoWrappedDataType => supportDataSchema(udt.toInternalType)

    case _ => false
  }

  override def newOutputFormat(
      path: String, dataSchema:
  RowType, taskId: Int): OutputFormat[BaseRow] = {
    val parquetOptions = (new TableProperties).putProperties(options).toKeyLowerCase
    val writeMode = FileSystemOptions.readWriteMode(parquetOptions)
    if (writeMode.equals(WriteMode.OVERWRITE)) {
      FileUtil.fullyDelete(new File(path))
    }
    val p = new Path(path)
    if (p.getFileSystem.exists(p) && !p.getFileSystem.getFileStatus(p).isDir) {
      throw new RuntimeException( "output dir [" + path + "] already existed.")
    }

    val blockSize = parquetOptions.getInteger(ParquetOptions.BLOCK_SIZE)

    val enableDictionary = parquetOptions.getBoolean(ParquetOptions.DICTIONARY_ENABLED)

    val compression = CompressionCodecName.valueOf(
      parquetOptions.getString(ParquetOptions.COMPRESSION_CODEC_NAME))

    new RowParquetOutputFormat(path,
      dataSchema.getFieldInternalTypes,
      dataSchema.getFieldNames,
      compression,
      blockSize,
      enableDictionary,
      false)
  }

  override def toString: String = "PARQUET"
}
