/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.sinks.parquet;

import org.apache.flink.core.fs.Path;
import org.apache.flink.table.api.VirtualColumn;
import org.apache.flink.table.dataformat.BaseRow;
import org.apache.flink.table.dataformat.BinaryString;
import org.apache.flink.table.types.InternalType;
import org.apache.flink.util.Preconditions;

import org.apache.parquet.hadoop.metadata.CompressionCodecName;

import java.io.IOException;

/**
 * For parquet table sink only when delete dml.
 */
public class DeleteRowParquetOutputFormat extends RowParquetOutputFormat {

	public static final String DELETE_DIR_SUFFIX = "_delete";
	private final int rowIdIndex = 0;
	private final int inputFileNameIndex = 1;
	private transient BinaryString lastInputFileName = null;
	private transient BinaryString currentInputFileName = null;
	private int taskNumber;
	private int numTasks;

	public DeleteRowParquetOutputFormat(String dir,
			InternalType[] fieldTypes,
			String[] fieldNames, CompressionCodecName compression, int blockSize, boolean enableDictionary) {
		super(dir, new InternalType[]{VirtualColumn.ROWID.getInternalType()}, new String[]{VirtualColumn.ROWID.getName()}, compression, blockSize, enableDictionary, false);
		Preconditions.checkArgument(fieldNames.length == 2,
				"Delete outputFormat should receive only " + VirtualColumn.ROWID.getName()
						+ " and " + VirtualColumn.FILENAME.getName());
		Preconditions.checkArgument(fieldNames[rowIdIndex].equals(VirtualColumn.ROWID.getName()), VirtualColumn.ROWID.getName() + " should be first.");
		Preconditions.checkArgument(fieldNames[inputFileNameIndex].equals(VirtualColumn.FILENAME.getName()), VirtualColumn.FILENAME.getName() + " should be second.");
	}

	@Override
	public void open(int taskNumber, int numTasks) throws IOException {
		this.taskNumber = taskNumber;
		this.numTasks = numTasks;
	}

	@Override
	public void writeRecord(BaseRow record) throws IOException {
		currentInputFileName = record.getBinaryString(inputFileNameIndex);
			if (lastInputFileName == null || currentInputFileName.compareTo(lastInputFileName) != 0) {
				lastInputFileName = currentInputFileName.copy();
				String writeFileName = FILE_PREFIX_NAME + numTasks + "-" + taskNumber + ".parquet";
				Path inputPath = new Path(lastInputFileName.toString());
				Path path = new Path(inputPath.suffix(DELETE_DIR_SUFFIX).toString() + "/" + initTime, writeFileName);
				makeRealWriter(path);
			}
		try {
			realWriter.write(null, record);
		} catch (final InterruptedException e) {
			throw new IOException(e);
		}
	}
}
