/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.sources.parquet.update;

import org.apache.flink.core.fs.FileStatus;
import org.apache.flink.core.fs.Path;
import org.apache.flink.table.api.VirtualColumn;
import org.apache.flink.table.sources.parquet.VectorizedColumnRowInputParquetFormat;
import org.apache.flink.table.types.InternalType;

import org.apache.hadoop.mapreduce.RecordReader;
import org.apache.parquet.filter2.predicate.FilterPredicate;

import java.util.ArrayList;
import java.util.List;

import static org.apache.flink.table.sinks.parquet.DeleteRowParquetOutputFormat.DELETE_DIR_SUFFIX;

/**
 * Parquet inputFormat for supported parquet files.
 */
public class UpdateVectorizedColumnRowInputParquetFormat extends VectorizedColumnRowInputParquetFormat {

	public UpdateVectorizedColumnRowInputParquetFormat(Path filePath,
			InternalType[] fieldTypes, String[] fieldNames, long limit) {
		super(filePath, fieldTypes, fieldNames, limit);
	}

	@Override
	protected RecordReader createReader(FilterPredicate filter) {
		List<String> filterNames = new ArrayList<>();
		List<InternalType> filterTypes = new ArrayList<>();

		boolean isDeleteSource = false;
		for (int i = 0; i < fieldNames.length; i++) {
			if (!fieldNames[i].equals(VirtualColumn.FILENAME.getName())) {
				filterNames.add(fieldNames[i]);
				filterTypes.add(fieldTypes[i]);
			} else {
				isDeleteSource = true;
			}
		}
		if (!isDeleteSource) {
			// append row_id column.
			filterNames.add(VirtualColumn.ROWID.getName());
			filterTypes.add(VirtualColumn.ROWID.getInternalType());
		}
		return new UpdateParquetVectorizedColumnRowReader(filterTypes.toArray(new InternalType[]{}), filterNames.toArray(new String[]{}), limit, isDeleteSource);
	}

	@Override
	public boolean acceptFile(FileStatus fileStatus) {
		return super.acceptFile(fileStatus) && !fileStatus.getPath().getName().endsWith(DELETE_DIR_SUFFIX);
	}
}
