/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
/*
 * Copyright 2005 Sun Microsystems, Inc. All rights reserved.
 */
/*
 * $Id: DOMManifest.java 1634871 2014-10-28 13:13:52Z mullan $
 */
package org.apache.jcp.xml.dsig.internal.dom;

import javax.xml.crypto.*;
import javax.xml.crypto.dsig.*;

import java.security.Provider;
import java.util.*;

import org.w3c.dom.Element;

/**
 * DOM-based implementation of Manifest.
 *
 * @author Sean Mullan
 */
public final class DOMManifest extends BaseStructure implements Manifest {

    private final List<Reference> references;
    private final String id;

    /**
     * Creates a <code>DOMManifest</code> containing the specified
     * list of {@link Reference}s and optional id.
     *
     * @param references a list of one or more <code>Reference</code>s. The list
     *    is defensively copied to protect against subsequent modification.
     * @param id the id (may be <code>null</code>
     * @throws NullPointerException if <code>references</code> is
     *    <code>null</code>
     * @throws IllegalArgumentException if <code>references</code> is empty
     * @throws ClassCastException if <code>references</code> contains any
     *    entries that are not of type {@link Reference}
     */
    public DOMManifest(List<DOMReference> references, String id) {
        if (references == null) {
            throw new NullPointerException("references cannot be null");
        }
        this.references =
            Collections.unmodifiableList(new ArrayList<Reference>(references));
        if (this.references.isEmpty()) {
            throw new IllegalArgumentException("list of references must " +
                "contain at least one entry");
        }
        for (int i = 0, size = this.references.size(); i < size; i++) {
            if (!(this.references.get(i) instanceof Reference)) {
                throw new ClassCastException
                    ("references["+i+"] is not a valid type");
            }
        }
        this.id = id;
    }

    /**
     * Creates a <code>DOMManifest</code> from an element.
     *
     * @param manElem a Manifest element
     */
    public DOMManifest(Element manElem, XMLCryptoContext context,
                       Provider provider)
        throws MarshalException
    {
        this.id = DOMUtils.getIdAttributeValue(manElem, "Id");
        
        boolean secVal = Utils.secureValidation(context);
        
        Element refElem = DOMUtils.getFirstChildElement(manElem, "Reference", XMLSignature.XMLNS);
        List<Reference> refs = new ArrayList<Reference>();
        refs.add(new DOMReference(refElem, context, provider));
        
        refElem = DOMUtils.getNextSiblingElement(refElem);
        while (refElem != null) {
            String localName = refElem.getLocalName();
            String namespace = refElem.getNamespaceURI();
            if (!localName.equals("Reference") || !XMLSignature.XMLNS.equals(namespace)) {        
                throw new MarshalException("Invalid element name: " +
                                           namespace + ":" + localName + ", expected Reference");
            }
            refs.add(new DOMReference(refElem, context, provider));
            if (secVal && refs.size() > DOMSignedInfo.MAXIMUM_REFERENCE_COUNT) {
                String error = "A maxiumum of " + DOMSignedInfo.MAXIMUM_REFERENCE_COUNT + " " 
                    + "references per Manifest are allowed with secure validation";
                throw new MarshalException(error);
            }
            refElem = DOMUtils.getNextSiblingElement(refElem);
        }
        this.references = Collections.unmodifiableList(refs);
    }

    @Override
    public String getId() {
        return id;
    }
    
    @SuppressWarnings("unchecked")
    public static List<Reference> getManifestReferences(Manifest mf) {
        return mf.getReferences();
    }
    
    @Override
    public List<Reference> getReferences() {
        return references;
    }

    public static void marshal(XmlWriter xwriter, Manifest manif, String dsPrefix, XMLCryptoContext context)
    throws MarshalException {
        xwriter.writeStartElement(dsPrefix, "Manifest", XMLSignature.XMLNS);
        xwriter.writeIdAttribute("", "", "Id", manif.getId());

        // add references
        @SuppressWarnings("unchecked")
        List<Reference> references = manif.getReferences();
        for (Reference ref : references) {
            ((DOMReference)ref).marshal(xwriter, dsPrefix, context);
        }
        xwriter.writeEndElement(); // "Manifest"
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }

        if (!(o instanceof Manifest)) {
            return false;
        }
        Manifest oman = (Manifest)o;

        boolean idsEqual = id == null ? oman.getId() == null
                                       : id.equals(oman.getId());

        return idsEqual && references.equals(oman.getReferences());
    }
    
    @Override
    public int hashCode() {
        int result = 17;
        if (id != null) {
            result = 31 * result + id.hashCode();
        }
        result = 31 * result + references.hashCode();
        
        return result;
    }
}
