/*
 * Decompiled with CFR 0.152.
 */
package org.influxdb.impl;

import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.function.BiConsumer;
import org.influxdb.InfluxDB;
import org.influxdb.InfluxDBException;
import org.influxdb.dto.BatchPoints;
import org.influxdb.dto.Point;
import org.influxdb.impl.BatchWriter;

class RetryCapableBatchWriter
implements BatchWriter {
    private InfluxDB influxDB;
    private BiConsumer<Iterable<Point>, Throwable> exceptionHandler;
    private LinkedList<BatchPoints> batchQueue;
    private int requestActionsLimit;
    private int retryBufferCapacity;
    private int usedRetryBufferCapacity;

    RetryCapableBatchWriter(InfluxDB influxDB, BiConsumer<Iterable<Point>, Throwable> exceptionHandler, int retryBufferCapacity, int requestActionsLimit) {
        this.influxDB = influxDB;
        this.exceptionHandler = exceptionHandler;
        this.batchQueue = new LinkedList();
        this.retryBufferCapacity = retryBufferCapacity;
        this.requestActionsLimit = requestActionsLimit;
    }

    @Override
    public synchronized void write(Collection<BatchPoints> collection) {
        ListIterator batchQueueIterator = this.batchQueue.listIterator();
        while (batchQueueIterator.hasNext()) {
            BatchPoints entry = (BatchPoints)batchQueueIterator.next();
            WriteResult result = this.tryToWrite(entry);
            if (result.outcome == WriteResultOutcome.WRITTEN || result.outcome == WriteResultOutcome.FAILED_RETRY_IMPOSSIBLE) {
                batchQueueIterator.remove();
                this.usedRetryBufferCapacity -= entry.getPoints().size();
                if (result.outcome != WriteResultOutcome.FAILED_RETRY_IMPOSSIBLE) continue;
                this.exceptionHandler.accept(entry.getPoints(), result.throwable);
                continue;
            }
            for (BatchPoints batchPoints : collection) {
                this.addToBatchQueue(batchPoints);
            }
            return;
        }
        Iterator<BatchPoints> collectionIterator = collection.iterator();
        while (collectionIterator.hasNext()) {
            BatchPoints batchPoints = collectionIterator.next();
            WriteResult result = this.tryToWrite(batchPoints);
            switch (result.outcome) {
                case FAILED_RETRY_POSSIBLE: {
                    this.addToBatchQueue(batchPoints);
                    while (collectionIterator.hasNext()) {
                        this.addToBatchQueue(collectionIterator.next());
                    }
                    break;
                }
                case FAILED_RETRY_IMPOSSIBLE: {
                    this.exceptionHandler.accept(batchPoints.getPoints(), result.throwable);
                    break;
                }
            }
        }
    }

    @Override
    public synchronized void close() {
        for (BatchPoints points : this.batchQueue) {
            WriteResult result = this.tryToWrite(points);
            if (result.outcome == WriteResultOutcome.WRITTEN) continue;
            this.exceptionHandler.accept(points.getPoints(), result.throwable);
        }
    }

    private WriteResult tryToWrite(BatchPoints batchPoints) {
        try {
            this.influxDB.write(batchPoints);
            return WriteResult.WRITTEN;
        }
        catch (InfluxDBException e) {
            return new WriteResult(e);
        }
        catch (Exception e) {
            return new WriteResult(WriteResultOutcome.FAILED_RETRY_POSSIBLE, e);
        }
    }

    private void evictTooOldFailedWrites() {
        while (this.usedRetryBufferCapacity > this.retryBufferCapacity && this.batchQueue.size() > 0) {
            List<Point> points = this.batchQueue.removeFirst().getPoints();
            this.usedRetryBufferCapacity -= points.size();
            this.exceptionHandler.accept(points, new InfluxDBException.RetryBufferOverrunException("Retry buffer overrun, current capacity: " + this.retryBufferCapacity));
        }
    }

    private void addToBatchQueue(BatchPoints batchPoints) {
        boolean hasBeenMergedIn;
        BatchPoints last;
        if (this.batchQueue.size() > 0 && (last = this.batchQueue.getLast()).getPoints().size() + batchPoints.getPoints().size() <= this.requestActionsLimit && (hasBeenMergedIn = last.mergeIn(batchPoints))) {
            return;
        }
        this.batchQueue.add(batchPoints);
        this.usedRetryBufferCapacity += batchPoints.getPoints().size();
        this.evictTooOldFailedWrites();
    }

    private static final class WriteResult {
        static final WriteResult WRITTEN = new WriteResult(WriteResultOutcome.WRITTEN);
        WriteResultOutcome outcome;
        Throwable throwable;

        private WriteResult(WriteResultOutcome outcome) {
            this.outcome = outcome;
        }

        private WriteResult(WriteResultOutcome outcome, Throwable throwable) {
            this.outcome = outcome;
            this.throwable = throwable;
        }

        private WriteResult(InfluxDBException e) {
            this.throwable = e;
            this.outcome = e.isRetryWorth() ? WriteResultOutcome.FAILED_RETRY_POSSIBLE : WriteResultOutcome.FAILED_RETRY_IMPOSSIBLE;
        }
    }

    private static enum WriteResultOutcome {
        WRITTEN,
        FAILED_RETRY_POSSIBLE,
        FAILED_RETRY_IMPOSSIBLE;

    }
}

